const express = require("express");
const JSZip = require("jszip");
const router = express.Router();
const axios = require("axios");
const fs = require("fs");
const { Parser } = require('json2csv');
const multer  = require('multer')
const path = require('path');
// const upload = multer({ dest: 'task_images/' })
const jwt = require('jsonwebtoken');
const secretKey = 'your_secret_key';

const {
  createCategory,
  updateCategory,
  deleteCategory,
  getCategories,
  getCategoryById,
  getCategoriesByCreatedBy,
} = require("../controllers/category");
const {
  createReport,
  updateReport,
  deleteReport,
  getReports,
  getReportById,
  getReportsByCreatedBy,
} = require("../controllers/reports");
const {
  createGroup,
  getGroupById,
  getGroupsByUserId,
  updateGroupById,
  softDeleteGroupById,
  getAllGroups,
} = require("../controllers/settings/groups");
const {
  createDevice,
  getDeviceById,
  getDevicesByUserId,
  softDeleteDeviceById,
  updateDeviceById,
  getAllDevices,
  findDeviceByFlespiIdAndPassword,
  findDeviceByUniqueIdAndPassword,
} = require("../controllers/settings/devices");
const { getGMTTimestampsForDay, formatDDMY, formatTimestamp, formatTimeISO, formatTime, formatTimeISOGMTMINUS5 } = require("../utils/commons");
const {
  createUser,
  getAllUsers,
  getUserById,
  getUsersByUserId,
  softDeleteUserById,
  updateUserById,
} = require("../controllers/settings/users");
const {
  IOU_1674670,
  IOU_1674833,
  IOU_1675142,
} = require("../controllers/reports/reports");
const {
  createUFT,
  getTokenByUserId,
  createUFT2,
} = require("../controllers/settings/usersTokens");
const { S3, ListObjectsV2Command } = require("@aws-sdk/client-s3");
const { createGunzip } = require("zlib");
const {
  createSnapshot,
  getAllSnapshots,
  getSnapshotsByUserId,
  getSnapshotsByFlespiIds,
  addSnapshotsDevices,
  getAllIdsAndNames,
  removeDevicesByIds,
  createSnapshot2,
  getAllSnapshots2,
  getSnapshots2ByFlespiIds,
} = require("../controllers/settings/snapshots");

const schedule = require("node-schedule");
const {
  createGeofence,
  getAllGeofences,
  getGeofenceById,
  getGeofencesByUserId,
  updateGeofenceById,
  softDeleteGeofenceById,
} = require("../controllers/settings/geofences");
const {
  createCalendar,
  getAllCalendars,
  getCalendarById,
  getCalendarsByUserId,
  updateCalendarById,
  softDeleteCalendarById,
} = require("../controllers/settings/calendars");

const {
  createAttribute,
  getAllAttributes,
  getAttributeById,
  getAttributesByUserId,
  updateAttributeById,
  softDeleteAttributeById,
} = require("../controllers/settings/attributesComputed");

const {
  createCommand,
  getAllCommands,
  getCommandById,
  getCommandsByUserId,
  updateCommandById,
  softDeleteCommandById,
} = require("../controllers/settings/commands");

const {
  createMaintenance,
  getAllMaintenances,
  getMaintenanceById,
  getMaintenancesByUserId,
  updateMaintenanceById,
  softDeleteMaintenanceById,
} = require("../controllers/settings/maintenance");
const {
  createDriver,
  getAllDrivers,
  getDriverById,
  getDriversByUserId,
  updateDriverById,
  softDeleteDriverById,
} = require("../controllers/settings/drivers");
const {
  createNotifications,
  getAllNotifications,
  getNotificationById,
  getNotificationsByUserId,
  updateNotificationById,
  softDeleteNotificationById,
} = require("../controllers/settings/notifications");
const { createCalculator, getAllCalculators, getCalculatorById, getCalculatorsByUserId, softDeleteCalculatorById, updateCalculatorById } = require("../controllers/settings/calculators");
const { createTask, getAllTasks, getTaskById, getTasksByUserId, getTasksByDeviceId, updateTaskStatusByIdAndDeviceId, createTaskSH, getTasksSHByUserId, createPointSH, getPointsSHByUserId } = require("../controllers/settings/tasks");


// const { createNotifications } = require("../controllers/settings/notifications");

const traccarBearerToken = process.env.TraccarToken;
const traccarApiUrl = `http://${process.env.TraccarPort}/api`;
const flespiToken = process.env.FlespiToken;
const flespiApiUrl = `https://flespi.io/gw`;

const baseURL = process.env.PRODUCTION_URL;

// _____________________________________________________________________________________________________________________

router.post("/category", async (req, res) => {
  try {
    const data = req.body;
    const insertID = await createCategory(data);

    data.id = insertID;

    res.status(201).json({
      status: true,
      message: "Category created successfully",
      data: data,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/category/:id", async (req, res) => {
  try {
    const categoryId = req.params.id;
    const data = req.body;

    await updateCategory(categoryId, data);

    res.status(200).json({
      status: true,
      message: "Category updated successfully",
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.delete("/category/:id", async (req, res) => {
  try {
    const categoryId = req.params.id;

    await deleteCategory(categoryId);

    res.status(200).json({
      status: true,
      message: "Category deleted successfully",
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/categories", async (req, res) => {
  try {
    const categories = await getCategories();

    res.status(200).json({
      status: true,
      data: categories,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/category/:id", async (req, res) => {
  try {
    const categoryId = req.params.id;
    const category = await getCategoryById(categoryId);

    res.status(200).json({
      status: true,
      data: category,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/categories/createdby/:createdBy", async (req, res) => {
  try {
    const createdBy = req.params.createdBy;
    const categories = await getCategoriesByCreatedBy(createdBy);

    res.status(200).json({
      status: true,
      data: categories,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

// _____________________________________________________________________________________________________________________

router.post("/report", async (req, res) => {
  try {
    const data = req.body;
    const insertID = await createReport(data);

    data.id = insertID;

    res.status(201).json({
      status: true,
      message: "Report created successfully",
      data: data,
    });
  } catch (error) {
    if (error.code === "ER_NO_REFERENCED_ROW_2") {
      res.status(400).json({
        status: false,
        error: "Unknow Category ID",
      });
    } else {
      console.error(error);
      res.status(500).json({
        status: false,
        error: "Internal Server Error",
      });
    }
  }
});

router.put("/report/:id", async (req, res) => {
  try {
    const reportId = req.params.id;
    const data = req.body;

    await updateReport(reportId, data);

    res.status(200).json({
      status: true,
      message: "Report updated successfully",
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.delete("/report/:id", async (req, res) => {
  try {
    const reportId = req.params.id;

    await deleteReport(reportId);

    res.status(200).json({
      status: true,
      message: "Report deleted successfully",
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/reports", async (req, res) => {
  try {
    const reports = await getReports();

    res.status(200).json({
      status: true,
      data: reports,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/report/:id", async (req, res) => {
  try {
    const reportId = req.params.id;
    const report = await getReportById(reportId);

    res.status(200).json({
      status: true,
      data: report,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/reports/createdby/:createdBy", async (req, res) => {
  try {
    const createdBy = req.params.createdBy;
    const reports = await getReportsByCreatedBy(createdBy);

    res.status(200).json({
      status: true,
      data: reports,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/device-type", async (req, res) => {
  try {
    const apiResponse = await axios.get(
      "https://flespi.io/gw/channel-protocols/all/device-types/all?fields=protocol_id%2Cprotocol_name%2Cid%2Cname%2Ctitle",
      {
        headers: {
          Authorization: `FlespiToken ${process.env.FlespiToken}`,
        },
      }
    );

    if (apiResponse.status === 200) {
      res.status(200).json({
        status: true,
        data: apiResponse.data.result,
      });
    } else {
      console.error("API request failed:", apiResponse.statusText);
      res.status(500).json({
        status: false,
        error: "Internal Server Error",
      });
    }
  } catch (error) {
    console.error("An error occurred while fetching data:", error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

// _____________________________________________________________________________________________________________________

router.get("/c-report/calcs/:cals/devices/:devices", async (req, res) => {
  try {
    const { cals, devices } = req.params;

    const validCals = [
      "1674670",
      "1674971",
      "1675136",
      "1674836",
      "1675142",
      "1674833",
    ];

    if (!validCals.includes(cals)) {
      return res.status(400).json({
        status: false,
        error: `Invalid calc id: ${cals}`,
      });
    }

    const timestamps =
      // cals === "1675142" || cals === "1674833"
      cals === "1674836" || cals === "1675142" || cals === "1674833"
        ? getGMTTimestampsForDay("dayBefore")
        : getGMTTimestampsForDay("today");
    // console.log

    const { startOfDay: begin, endOfDay: end } = timestamps;

    // const apiUrl = `https://flespi.io/gw/calcs/${cals}/devices/${devices}/intervals/all`;
    const apiUrl = `https://flespi.io/gw/calcs/${cals}/devices/${devices}/intervals/all?data=%7B%22begin%22%3A${begin}%2C%22end%22%3A${end}%7D`;

    const headers = {
      Authorization: `FlespiToken ${flespiToken}`,
    };

    const response = await axios.get(apiUrl, { headers });

    res.status(200).json({
      status: true,
      // cals: cals,
      data: response.data.result,
    });
  } catch (error) {
    console.error("Unexpected error:", error);

    // Include more details about the error in the response
    res.status(500).json({
      status: false,
      error: `An error occurred while fetching data: ${error.message}`,
    });
  }
});

router.get(
  "/c-report/calcs/:cals/devices/:devices/begin/:begin/end/:end",
  async (req, res) => {
    try {
      const { cals, devices, begin, end } = req.params;

      const validCals = [
        "1674670",
        "1674971",
        "1675136",
        "1674836",
        "1675142",
        "1674833",
      ];

      if (!validCals.includes(cals)) {
        return res.status(400).json({
          status: false,
          error: `Invalid calc id: ${cals}`,
        });
      }

      // const timestamps =
      //   cals === "1675142" || cals === "1674833"
      //     ?
      //     getGMTTimestampsForDay("dayBefore")
      //     :
      //     getGMTTimestampsForDay("today")
      //     ;

      // console.log

      // const { startOfDay: begin, endOfDay: end } = timestamps;

      // const apiUrl = `https://flespi.io/gw/calcs/${cals}/devices/${devices}/intervals/all`;
      const apiUrl = `https://flespi.io/gw/calcs/${cals}/devices/${devices}/intervals/all?data=%7B%22begin%22%3A${begin}%2C%22end%22%3A${end}%7D`;

      const headers = {
        Authorization: `FlespiToken ${flespiToken}`,
      };

      const response = await axios.get(apiUrl, { headers });

      res.status(200).json({
        status: true,
        // cals: cals,
        data: response.data.result,
      });
    } catch (error) {
      console.error("Unexpected error:", error);

      // Include more details about the error in the response
      res.status(500).json({
        status: false,
        error: `An error occurred while fetching data: ${error.message}`,
      });
    }
  }
);

router.post("/c-report/calcs/:cals/devices/:devices", async (req, res) => {
  try {
    const { cals, devices } = req.params;

    const validCals = ["1674670", "1675142", "1674833"];

    if (!validCals.includes(cals)) {
      return res.status(400).json({
        status: false,
        error: `Invalid calc id: ${cals}`,
      });
    }

    const timestamps =
      // cals === "1675142" || cals === "1674833"
      cals === "1674836" || cals === "1675142" || cals === "1674833"
        ? getGMTTimestampsForDay("dayBefore")
        : // ? getGMTTimestampsForDay("today")
          getGMTTimestampsForDay("today");

    const { startOfDay: begin, endOfDay: end } = timestamps;

    const apiUrl = `https://flespi.io/gw/calcs/${cals}/devices/${devices}/intervals/all?data=%7B%22begin%22%3A${begin}%2C%22end%22%3A${end}%7D`;

    const headers = {
      Authorization: `FlespiToken ${flespiToken}`,
    };

    const response = await axios.get(apiUrl, { headers });

    const ioFunctionMap = {
      1675142: IOU_1675142,
      1674833: IOU_1674833,
      1674670: IOU_1674670,
    };

    const ioFunction = ioFunctionMap[cals];

    if (!ioFunction) {
      return res.status(400).json({
        status: false,
        error: `This calc:${cals} is not valid of post.`,
      });
    }

    let insertedCount = 0;
    let updatedCount = 0;

    for (const reportData of response.data.result) {
      const insert = await ioFunction(reportData);
      insert > 0 ? insertedCount++ : updatedCount++;
    }

    res.status(200).json({
      status: true,
      insertedCount,
      updatedCount,
      message: "Data Inserted Successfully",
    });
  } catch (error) {
    console.error("Unexpected error:", error);

    res.status(500).json({
      status: false,
      error: `An error occurred while fetching data: ${error.message}`,
    });
  }
});

router.get("/flespi-devices", async (req, res) => {
  try {
    const apiUrl = `https://flespi.io/gw/devices/all?fields=configuration%2Cdevice_type_name%2Cprotocol_name%2Cdevice_type_id%2Cprotocol_id%2Cid%2Cname`;

    const headers = {
      Authorization: `FlespiToken ${flespiToken}`,
    };

    const response = await axios.get(apiUrl, { headers });

    res.status(200).json({
      status: true,
      data: response.data.result,
    });
  } catch (error) {
    console.error("Unexpected error:", error);

    // Include more details about the error in the response
    res.status(500).json({
      status: false,
      error: `An error occurred while fetching data: ${error.message}`,
    });
  }
});

router.get("/flespi-calcs", async (req, res) => {
  try {
    const apiUrl = `https://flespi.io/gw/calcs/all?fields=id%2Cname`;

    const headers = {
      Authorization: `FlespiToken ${flespiToken}`,
    };

    const response = await axios.get(apiUrl, { headers });

    res.status(200).json({
      status: true,
      data: response.data.result,
    });
  } catch (error) {
    console.error("Unexpected error:", error);

    // Include more details about the error in the response
    res.status(500).json({
      status: false,
      error: `An error occurred while fetching data: ${error.message}`,
    });
  }
});

router.get("/flespi-device-types", async (req, res) => {
  try {
    const apiUrl = `https://flespi.io/gw/channel-protocols/all/device-types/all?fields=id%2Cprotocol_id%2Cprotocol_name%2Cname%2Ctitle`;

    const headers = {
      Authorization: `FlespiToken ${flespiToken}`,
    };

    const response = await axios.get(apiUrl, { headers });

    res.status(200).json({
      status: true,
      data: response.data.result,
    });
  } catch (error) {
    console.error("Unexpected error:", error);

    // Include more details about the error in the response
    res.status(500).json({
      status: false,
      error: `An error occurred while fetching data: ${error.message}`,
    });
  }
});

router.get("/calcs/:cals/devices/:devices/intervals/:intervals", async (req, res) => {
  const { cals, devices, intervals } = req.params;
  const queryString = req.url.split('?')[1];
  const data = queryString ? queryString.split('=')[1] : '';
  const apiUrl = `https://flespi.io/gw/calcs/${cals}/devices/${devices}/intervals/${intervals}${data ? `?data=${data}` : ''}`;
  const headers = {
    Authorization: `FlespiToken ${flespiToken}`,
  };

  try {
    const response = await axios.get(apiUrl, { headers });
    res.status(200).json({
      status: true,
      data: response.data.result,
    });
  } catch (error) {
    if (error.response?.data?.errors && Array.isArray(error.response?.data?.errors)) {
      res.status(error.response.status).json({
        status: false,
        errors: error.response?.data?.errors,
      });
    } else {
      res.status(500).json({
        status: false,
        message: `Error: ${error.message}`,
      });
    }
  }
});

// ______________________________________________________________________________________________________________________

router.post("/new-groups", async (req, res) => {
  try {
    const traccarRequestData = {
      name: req.body.name,
    };

    if (req.body.attributes) {
      traccarRequestData.attributes = req.body.attributes;
    }

    if (req.body.groupId) {
      traccarRequestData.groupId = req.body.groupId;
    }

    const flespiRequestData = {
      name: req.body.name,
      item_type: 11,
      metadata: {},
    };

    if (req.body.groupId) {
      flespiRequestData.metadata.groupId = req.body.groupId;
    }

    if (req.body.attributes) {
      flespiRequestData.metadata = {
        ...flespiRequestData.metadata,
        ...req.body.attributes,
      };
    }

    const [traccarResponse, flespiResponse] = await Promise.all([
      axios.post(`${traccarApiUrl}/groups`, traccarRequestData, {
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }),
      axios.post(
        `${flespiApiUrl}/groups?fields=id%2Cname%2Cmetadata`,
        [flespiRequestData],
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: flespiToken,
          },
        }
      ),
    ]);

    const traccarResponsePermissions = await axios.post(
      `${traccarApiUrl}/permissions`,
      { userId: req.body.userId, groupId: traccarResponse.data.id },
      {
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    let responses = {
      userId: req.body.userId,
      traccar: traccarResponse.data,
      flespi: flespiResponse.data.result[0],
    };

    const insertID = await createGroup(responses);

    responses.id = insertID;

    res.status(200).json({
      status: true,
      message: "Group created successfully",
      data: responses,
    });
  } catch (error) {
    console.error(
      "Error in creating group:",
      error.response ? error.response.data : error.message
    );

    let errorMessage = "Internal Server Error";
    if (error.response && error.response.status === 400) {
      errorMessage = "Bad Request: Invalid data provided";
    } else if (error.response && error.response.status === 401) {
      errorMessage = "Unauthorized";
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.get("/new-groups/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const groups = await getGroupsByUserId(userId);

    res.status(200).json({
      status: true,
      data: groups,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-groups", async (req, res) => {
  try {
    const group = await getAllGroups();

    if (!group) {
      res.status(404).json({
        status: false,
        error: "Group not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: group,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-groups/:id", async (req, res) => {
  try {
    const groupId = req.params.id;
    const group = await getGroupById(groupId);

    if (!group) {
      res.status(404).json({
        status: false,
        error: "Group not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: group,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/new-groups/:id", async (req, res) => {
  try {
    const groupId = req.params.id;
    const group = await getGroupById(groupId);

    if (!group) {
      return res.status(404).json({
        status: false,
        error: "Group not found",
      });
    }

    const traccarRequestData = {
      name: req.body.name,
      id: group.traccarId,
    };

    if (req.body.attributes) {
      traccarRequestData.attributes = req.body.attributes;
    }

    if (req.body.groupId) {
      traccarRequestData.groupId = req.body.groupId;
    }

    const flespiRequestData = {
      name: req.body.name,
      metadata: {},
    };

    if (req.body.groupId) {
      flespiRequestData.metadata.groupId = req.body.groupId;
    }

    if (req.body.attributes) {
      flespiRequestData.metadata = {
        ...flespiRequestData.metadata,
        ...req.body.attributes,
      };
    }

    const [traccarResponse, flespiResponse] = await Promise.all([
      axios.put(
        `${traccarApiUrl}/groups/${group.traccarId}`,
        traccarRequestData,
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${traccarBearerToken}`,
          },
        }
      ),
      axios.put(
        `${flespiApiUrl}/groups/${group.flespiId}?fields=id%2Cname%2Cmetadata`,
        flespiRequestData,
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: flespiToken,
          },
        }
      ),
    ]);

    const responses = {
      traccar: traccarResponse.data,
      flespi: flespiResponse.data.result[0],
    };

    const update = await updateGroupById(groupId, responses);

    update.attributes = JSON.parse(update.attributes);

    res.status(200).json({
      status: true,
      data: update,
      message: "Group Updated successfully.",
    });
  } catch (error) {
    console.error(error);
    if (error.response && error.response.data && error.response.data.errors) {
      const serverErrors = error.response.data.errors;
      console.error("Server Errors:", serverErrors);
      res.status(400).json({
        status: false,
        error: "Bad Request",
        details: serverErrors,
      });
    } else {
      res.status(500).json({
        status: false,
        error: "Internal Server Error",
      });
    }
  }
});

router.delete("/new-groups/:id", async (req, res) => {
  try {
    const groupId = req.params.id;
    const group = await getGroupById(groupId);

    if (!group) {
      return res.status(404).json({
        status: false,
        error: "Group not found",
      });
    }

    const [traccarResponse, flespiResponse] = await Promise.allSettled([
      axios.delete(`${traccarApiUrl}/groups/${group.traccarId}`, {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }),
      axios.delete(`${flespiApiUrl}/groups/${group.flespiId}`, {
        headers: {
          Authorization: flespiToken,
        },
      }),
    ]);

    const failedRequests = [traccarResponse, flespiResponse].filter(
      (result) => result.status === "rejected"
    );

    if (failedRequests.length > 0) {
      const errorDetails = failedRequests.map(
        (result) => result.reason.message
      );
      console.error("Failed Requests:", errorDetails);
      return res.status(500).json({
        status: false,
        error: "Failed to delete from external services",
        details: errorDetails,
      });
    }

    const update = await softDeleteGroupById(groupId);

    res.status(200).json({
      status: true,
      message: "Group Deleted successfully.",
    });
  } catch (error) {
    console.error(error);

    if (error.response && error.response.data && error.response.data.errors) {
      const serverErrors = error.response.data.errors;
      console.error("Server Errors:", serverErrors);
      res.status(400).json({
        status: false,
        error: "Bad Request",
        details: serverErrors,
      });
    } else {
      res.status(500).json({
        status: false,
        error: "Internal Server Error",
      });
    }
  }
});

// ______________________________________________________________________________________________________________________

router.post("/new-devices", async (req, res) => {
  try {
    const requestData = req.body;
    const isSuperAdmin =
      typeof requestData.isSuperAdmin === "boolean"
        ? requestData.isSuperAdmin
        : undefined;

    // If isSuperAdmin is not provided or not a boolean, return an error
    if (isSuperAdmin === undefined) {
      return res.status(400).json({
        status: false,
        error: "isSuperAdmin field is required and must be a boolean.",
      });
    }

    // Initialize an array to store missing tokens
    const missingTokens = [];

    // If isSuperAdmin is false, check for the presence of flespiUserToken
    if (!isSuperAdmin && !requestData.flespiUserToken) {
      missingTokens.push("Flespi");
    }

    // If isSuperAdmin is false, check for the presence of traccarUserToken
    if (!isSuperAdmin && !requestData.traccarUserToken) {
      missingTokens.push("Nextop");
    }

    // If any token is missing, construct the error message accordingly
    if (missingTokens.length > 0) {
      let errorMessage = "Token(s) required for non-superAdmin users: ";
      errorMessage += missingTokens.join(", ");

      return res.status(400).json({
        status: false,
        error: errorMessage,
      });
    }

    const [traccarAllDevices, flespiAllDevices] = await Promise.all([
      axios.get(`${traccarApiUrl}/devices`, {
        headers: {
          "Content-Type": "application/json",
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }),
      axios.get(
        `${flespiApiUrl}/devices/all?fields=id%2Cname%2Cdevice_type_id%2Cdevice_type_name%2Cconfiguration`,
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: flespiToken,
          },
        }
      ),
    ]);

    function getDeviceDetails(uniqueId, traccarDevices, flespiDevices) {
      const traccarDevice = traccarDevices.find(
        (device) => device.uniqueId === uniqueId
      );
      const flespiDevice = flespiDevices.find(
        (device) => device.configuration.ident === uniqueId
      );

      return { traccarDevice, flespiDevice };
    }

    const { traccarDevice, flespiDevice } = getDeviceDetails(
      requestData?.uniqueId,
      traccarAllDevices?.data,
      flespiAllDevices?.data?.result
    );

    const requiredFields = ["userId", "name", "uniqueId", "device_type_id"];
    const missingFields = requiredFields.filter((field) => !requestData[field]);

    if (missingFields.length > 0) {
      return res.status(400).json({
        status: false,
        error: `Missing required fields: ${missingFields.join(", ")}`,
      });
    }

    const fieldsToIncludeTraccar = [
      "name",
      "uniqueId",
      "status",
      "disabled",
      "groupId",
      "phone",
      "model",
      "contact",
      "category",
      "attributes",
    ];

    const traccarRequestData = {};

    fieldsToIncludeTraccar.forEach((field) => {
      if (req.body[field] !== undefined) {
        traccarRequestData[field] = req.body[field];
      }
    });

    const fieldsToIncludeFlespi = [
      "name",
      "device_type_id",
      "configuration",
      "metadata",
      "media_ttl",
      "messages_ttl",
    ];

    const flespiRequestData = {};

    fieldsToIncludeFlespi.forEach((field) => {
      switch (field) {
        case "media_ttl":
          flespiRequestData[field] =
            req.body[field] !== undefined ? req.body[field] * 86400 : 259200;
          break;
        case "messages_ttl":
          flespiRequestData[field] =
            req.body[field] !== undefined ? req.body[field] * 86400 : 259200;
          break;
        case "configuration":
          flespiRequestData[field] = {
            ident: req.body.uniqueId || undefined,
          };
          break;
        case "metadata":
          const metadataFields = [
            "speedLimit",
            "fuelDropThreshold",
            "fuelIncreaseThreshold",
            "report.ignoreOdometer",
            "web.reportColor",
            "devicePassword",
            "deviceImage",
            "processing.copyAttributes",
            "decoder.timezone",
            "category",
            "groupId",
            "model",
            "contact",
            "phone",
            "expirationTime",
            "disabled",
            "attributes",
          ];
          flespiRequestData[field] = {};

          metadataFields.forEach((metadataField) => {
            if (req.body[metadataField] !== undefined) {
              flespiRequestData[field][metadataField] = req.body[metadataField];
            }
          });
          break;
        default:
          if (req.body[field] !== undefined) {
            flespiRequestData[field] = req.body[field];
          }
      }
    });

    if (traccarDevice && flespiDevice) {
      return res.status(400).json({
        status: false,
        error:
          "Device with this unique ID already exists in both Nextop and Flespi.",
        details: {
          traccarDevice,
          flespiDevice,
        },
        // data: responses,
      });
    } else if (traccarDevice) {
      return res.status(400).json({
        status: false,
        error: "Device with this unique ID already exists in Nextop.",
        details: {
          traccarDevice,
        },
        // data: responses,
      });
    } else if (flespiDevice) {

      const traccarResponse = await axios.post(
        `${traccarApiUrl}/devices`,
        traccarRequestData,
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: isSuperAdmin
              ? `Bearer ${traccarBearerToken}`
              : `Bearer ${requestData.traccarUserToken}`,
          },
        }
      );

      const traccarResponsePermissions = await axios.post(
        `${traccarApiUrl}/permissions`,
        { userId: req.body.userId, deviceId: traccarResponse.data.id },
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${traccarBearerToken}`,
          },
        }
      );

      let responses = {
        userId: req.body.userId,
        password: req.body.password,
        traccar: traccarResponse.data,
        flespi: flespiDevice,
      };

      const insertID = await createDevice(responses);

      responses.id = insertID;

      res.status(200).json({
        status: true,
        message:
          "device with this ident already exist in flespi, device request success in Nextop.",
        data: responses,
      });
    } else {

      const flespiResponse = await axios.post(
        `${flespiApiUrl}/devices?fields=id%2Cname%2Cconfiguration%2Cmetadata%2Cdevice_type_id%2Cdevice_type_name%2Cprotocol_id%2Cprotocol_name`,
        [flespiRequestData],
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: isSuperAdmin
              ? flespiToken
              : requestData.flespiUserToken,
          },
        }
      );

      const traccarResponse = await axios.post(
        `${traccarApiUrl}/devices`,
        traccarRequestData,
        {
          headers: {
            "Content-Type": "application/json",
            // Authorization: `Bearer ${traccarBearerToken}`,
            Authorization: isSuperAdmin
              ? `Bearer ${traccarBearerToken}`
              : `Bearer ${requestData.traccarUserToken}`,
          },
        }
      );

      const traccarResponsePermissions = await axios.post(
        `${traccarApiUrl}/permissions`,
        { userId: req.body.userId, deviceId: traccarResponse.data.id },
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${traccarBearerToken}`,
          },
        }
      );

      let responses = {
        userId: req.body.userId,
        password: req.body.password,
        traccar: traccarResponse.data,
        flespi: {
          ...flespiResponse.data.result[0],
          media_ttl: req.body.media_ttl !== undefined ? req.body.media_ttl : 3,
          messages_ttl:
            req.body.messages_ttl !== undefined ? req.body.messages_ttl : 3,
        },
      };

      const insertID = await createDevice(responses);

      responses.id = insertID;

      res.status(200).json({
        status: true,
        message: "device request success",
        data: responses,
      });
    }
  } catch (error) {
    if (error?.response?.data?.errors) {
      console.error(
        "Error in creating device response data errors:",
        error.response.data.errors
      );
    } else {
      console.error("Error in creating device:", error);
    }
    let errorMessage = "Internal Server Error";

    if (error.response) {
      const flespiErrors = error.response.data.errors;

      if (flespiErrors && flespiErrors.length > 0) {
        errorMessage = `Flespi Error: ${flespiErrors
          .map((err) => err.reason)
          .join(", ")}`;
      } else if (error.response.status === 400) {
        if (error.response.data.includes("Data truncation")) {
          errorMessage =
            "Nextop Error: Data truncation issue. Check date format and value.";
        } else if (error.response.data.includes("Duplicate entry")) {
          errorMessage =
            "Nextop Error: Duplicate entry issue. Choose a different unique identifier.";
        } else {
          errorMessage =
            "Nextop Error: Bad Request. Please check your request data.";
        }
      }
    } else {
      console.log("Unexpected error occurred. Please try again.");
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      errorMessage: errorMessage,
    });
  }
});

// router.post("/new-devices", async (req, res) => {
//   try {
//     const requestData = req.body;
//     const isSuperAdmin =
//       typeof requestData.isSuperAdmin === "boolean"
//         ? requestData.isSuperAdmin
//         : undefined;

//     // If isSuperAdmin is not provided or not a boolean, return an error
//     if (isSuperAdmin === undefined) {
//       return res.status(400).json({
//         status: false,
//         error: "isSuperAdmin field is required and must be a boolean.",
//       });
//     }

//     // Initialize an array to store missing tokens
//     const missingTokens = [];

//     // If isSuperAdmin is false, check for the presence of flespiUserToken
//     if (!isSuperAdmin && !requestData.flespiUserToken) {
//       missingTokens.push("Flespi");
//     }

//     // If isSuperAdmin is false, check for the presence of traccarUserToken
//     if (!isSuperAdmin && !requestData.traccarUserToken) {
//       missingTokens.push("Nextop");
//     }

//     // If any token is missing, construct the error message accordingly
//     if (missingTokens.length > 0) {
//       let errorMessage = "Token(s) required for non-superAdmin users: ";
//       errorMessage += missingTokens.join(", ");

//       return res.status(400).json({
//         status: false,
//         error: errorMessage,
//       });
//     }

//     const [traccarAllDevices, flespiAllDevices] = await Promise.all([
//       axios.get(`${traccarApiUrl}/devices`, {
//         headers: {
//           "Content-Type": "application/json",
//           Authorization: `Bearer ${traccarBearerToken}`,
//         },
//       }),
//       axios.get(
//         `${flespiApiUrl}/devices/all?fields=id%2Cname%2Cdevice_type_id%2Cdevice_type_name%2Cconfiguration`,
//         {
//           headers: {
//             "Content-Type": "application/json",
//             Authorization: flespiToken,
//           },
//         }
//       ),
//     ]);

//     function getDeviceDetails(uniqueId, traccarDevices, flespiDevices) {
//       const traccarDevice = traccarDevices.find(
//         (device) => device.uniqueId === uniqueId
//       );
//       const flespiDevice = flespiDevices.find(
//         (device) => device.configuration.ident === uniqueId
//       );

//       return { traccarDevice, flespiDevice };
//     }

//     const { traccarDevice, flespiDevice } = getDeviceDetails(
//       requestData?.uniqueId,
//       traccarAllDevices?.data,
//       flespiAllDevices?.data?.result
//     );

//     const requiredFields = ["userId", "name", "uniqueId", "device_type_id"];
//     const missingFields = requiredFields.filter((field) => !requestData[field]);

//     if (missingFields.length > 0) {
//       return res.status(400).json({
//         status: false,
//         error: `Missing required fields: ${missingFields.join(", ")}`,
//       });
//     }

//     const fieldsToIncludeTraccar = [
//       "name",
//       "uniqueId",
//       "status",
//       "disabled",
//       "groupId",
//       "phone",
//       "model",
//       "contact",
//       "category",
//       // "expirationTime",
//       "attributes",
//     ];

//     const traccarRequestData = {};

//     fieldsToIncludeTraccar.forEach((field) => {
//       if (req.body[field] !== undefined) {
//         traccarRequestData[field] = req.body[field];
//       }
//     });

//     const fieldsToIncludeFlespi = [
//       "name",
//       "device_type_id",
//       "configuration",
//       "metadata",
//       "media_ttl",
//       "messages_ttl",
//     ];

//     const flespiRequestData = {};

//     fieldsToIncludeFlespi.forEach((field) => {
//       switch (field) {
//         case "media_ttl":
//           flespiRequestData[field] =
//             req.body[field] !== undefined ? req.body[field] * 86400 : 259200;
//           break;
//         case "messages_ttl":
//           flespiRequestData[field] =
//             req.body[field] !== undefined ? req.body[field] * 86400 : 259200;
//           break;
//         case "configuration":
//           flespiRequestData[field] = {
//             ident: req.body.uniqueId || undefined,
//           };
//           break;
//         case "metadata":
//           const metadataFields = [
//             "speedLimit",
//             "fuelDropThreshold",
//             "fuelIncreaseThreshold",
//             "report.ignoreOdometer",
//             "web.reportColor",
//             "devicePassword",
//             "deviceImage",
//             "processing.copyAttributes",
//             "decoder.timezone",
//             "category",
//             "groupId",
//             "model",
//             "contact",
//             "phone",
//             "expirationTime",
//             "disabled",
//             "attributes",
//           ];
//           flespiRequestData[field] = {};

//           metadataFields.forEach((metadataField) => {
//             if (req.body[metadataField] !== undefined) {
//               flespiRequestData[field][metadataField] = req.body[metadataField];
//             }
//           });
//           break;
//         default:
//           if (req.body[field] !== undefined) {
//             flespiRequestData[field] = req.body[field];
//           }
//       }
//     });

//     if (traccarDevice && flespiDevice) {
//       return res.status(400).json({
//         status: false,
//         error:
//           "Device with this unique ID already exists in both Nextop and Flespi.",
//         details: {
//           traccarDevice,
//           flespiDevice,
//         },
//         // data: responses,
//       });
//     } else if (traccarDevice) {
//       return res.status(400).json({
//         status: false,
//         error: "Device with this unique ID already exists in Nextop.",
//         details: {
//           traccarDevice,
//         },
//         // data: responses,
//       });
//     } else if (flespiDevice) {
//       // return res.status(400).json({
//       //     status: false,
//       //     error: "Device with this unique ID already exists in Flespi.",
//       //     details: {
//       //         flespiDevice,
//       //     },
//       //     // data: responses,
//       // });
//       const traccarResponse = await axios.post(
//         `${traccarApiUrl}/devices`,
//         traccarRequestData,
//         {
//           headers: {
//             "Content-Type": "application/json",
//             Authorization: isSuperAdmin
//               ? `Bearer ${traccarBearerToken}`
//               : `Bearer ${requestData.traccarUserToken}`,
//           },
//         }
//       );

//       const traccarResponsePermissions = await axios.post(
//         `${traccarApiUrl}/permissions`,
//         { userId: req.body.userId, deviceId: traccarResponse.data.id },
//         {
//           headers: {
//             "Content-Type": "application/json",
//             Authorization: `Bearer ${traccarBearerToken}`,
//           },
//         }
//       );

//       let responses = {
//         userId: req.body.userId,

//         traccar: traccarResponse.data,
//         // traccar: traccarRequestData,
//         flespi: flespiDevice,
//       };

//       const insertID = await createDevice(responses);

//       responses.id = insertID;

//       res.status(200).json({
//         status: true,
//         message:
//           "device with this ident already exist in flespi, device request success in Nextop.",
//         data: responses,
//       });
//     } else {

      
//       const flespiResponse = await axios.post(
//         `${flespiApiUrl}/devices?fields=id%2Cname%2Cconfiguration%2Cmetadata%2Cdevice_type_id%2Cdevice_type_name%2Cprotocol_id%2Cprotocol_name`,
//         [flespiRequestData],
//         {
//           headers: {
//             "Content-Type": "application/json",
//             Authorization: isSuperAdmin
//               ? flespiToken
//               : requestData.flespiUserToken,
//           },
//         }
//       );

//       // console.log("flespiResponse", flespiResponse.data.result)

//       const traccarResponse = await axios.post(
//         `${traccarApiUrl}/devices`,
//         traccarRequestData,
//         {
//           headers: {
//             "Content-Type": "application/json",
//             // Authorization: `Bearer ${traccarBearerToken}`,
//             Authorization: isSuperAdmin
//               ? `Bearer ${traccarBearerToken}`
//               : `Bearer ${requestData.traccarUserToken}`,
//           },
//         }
//       );

//       // console.log("traccarResponse", traccarResponse.data)

//       // console.log(`${traccarApiUrl}/permissions`, { userId: req.body.userId, deviceId: traccarResponse.data.id })

//       const traccarResponsePermissions = await axios.post(
//         `${traccarApiUrl}/permissions`,
//         { userId: req.body.userId, deviceId: traccarResponse.data.id },
//         {
//           headers: {
//             "Content-Type": "application/json",
//             Authorization: `Bearer ${traccarBearerToken}`,
//           },
//         }
//       );


//       // console.log("traccarResponsePermissions", traccarResponsePermissions)

//       let responses = {
//         userId: req.body.userId,
//         traccar: traccarResponse.data,
//         // taca: traccarResponsePermissions.data,
//         // traccar: traccarRequestData,
//         // flespi: flespiRequestData,
//         flespi: {
//           ...flespiResponse.data.result[0],
//           media_ttl: req.body.media_ttl !== undefined ? req.body.media_ttl : 3,
//           messages_ttl:
//             req.body.messages_ttl !== undefined ? req.body.messages_ttl : 3,
//         },
//       };

//       const insertID = await createDevice(responses);

//       responses.id = insertID;

//       res.status(200).json({
//         status: true,
//         message: "device request success",
//         data: responses,
//       });
//     }
//   } catch (error) {
//     if (error?.response?.data?.errors) {
//       console.error(
//         "Error in creating device response data errors:",
//         error.response.data.errors
//       );
//     } else {
//       console.error("Error in creating device:", error);
//     }
//     let errorMessage = "Internal Server Error";

//     if (error.response) {
//       const flespiErrors = error.response.data.errors;

//       if (flespiErrors && flespiErrors.length > 0) {
//         // Flespi error handling
//         errorMessage = `Flespi Error: ${flespiErrors
//           .map((err) => err.reason)
//           .join(", ")}`;
//         // console.log();
//       } else if (error.response.status === 400) {
//         // Nextop error handling
//         if (error.response.data.includes("Data truncation")) {
//           errorMessage =
//             "Nextop Error: Data truncation issue. Check date format and value.";
//         } else if (error.response.data.includes("Duplicate entry")) {
//           errorMessage =
//             "Nextop Error: Duplicate entry issue. Choose a different unique identifier.";
//         } else {
//           errorMessage =
//             "Nextop Error: Bad Request. Please check your request data.";
//         }
//       }
//     } else {
//       console.log("Unexpected error occurred. Please try again.");
//     }

//     res.status(error.response ? error.response.status : 500).json({
//       status: false,
//       errorMessage: errorMessage,
//       // error: error,
//     });
//   }
// });

router.get("/new-devices", async (req, res) => {
  try {
    const devices = await getAllDevices();
    const groups = await getAllGroups();

    if (!devices) {
      res.status(404).json({
        status: false,
        error: "Device not found",
      });
      return;
    }

    const enrichedDevices = devices.map((device) => {
      const group = groups.find((group) => group.traccarId === device.groupId);
      return {
        ...device,
        groupName: group ? group.name : `Unknown (${device.groupId})`,
      };
    });

    res.status(200).json({
      status: true,
      data: enrichedDevices,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-devices/:id", async (req, res) => {
  try {
    const deviceId = req.params.id;
    const device = await getDeviceById(deviceId);

    if (!device) {
      res.status(404).json({
        status: false,
        error: "Device not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: device,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-devices/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const devices = await getDevicesByUserId(userId);
    const groups = await getAllGroups();

    if (!devices) {
      res.status(404).json({
        status: false,
        error: "Device not found",
      });
      return;
    }

    const enrichedDevices = devices.map((device) => {
      const group = groups.find((group) => group.traccarId === device.groupId);
      return {
        ...device,
        groupName: group ? group.name : `Unknown (${device.groupId})`,
      };
    });


    res.status(200).json({
      status: true,
      data: enrichedDevices,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/new-devices/:id", async (req, res) => {
  try {
    const deviceId = req.params.id;
    const device = await getDeviceById(deviceId);
    if (!device) {
      return res.status(404).json({
        status: false,
        error: "Device not found",
      });
    }
    const requestData = req.body;

    const requiredFields = ["userId", "name", "uniqueId", "device_type_id"];
    const missingFields = requiredFields.filter((field) => !requestData[field]);

    if (missingFields.length > 0) {
      return res.status(400).json({
        status: false,
        error: `Missing required fields: ${missingFields.join(", ")}`,
      });
    }

    const fieldsToIncludeTraccar = [
      "id",
      "name",
      "uniqueId",
      "status",
      "disabled",
      "groupId",
      "phone",
      "model",
      "contact",
      "category",
      // "expirationTime",
      "attributes",
    ];

    const traccarRequestData = {};

    fieldsToIncludeTraccar.forEach((field) => {
      const requestField = field === "id" ? "traccarId" : field;
      if (req.body[requestField] !== undefined) {
        traccarRequestData[field] = req.body[requestField];
      }
    });

    const fieldsToIncludeFlespi = [
      "name",
      "device_type_id",
      "configuration",
      "metadata",
      "media_ttl",
      "messages_ttl",
    ];

    const flespiRequestData = {};

    fieldsToIncludeFlespi.forEach((field) => {
      switch (field) {
        case "media_ttl":
          flespiRequestData[field] =
            req.body[field] !== undefined ? req.body[field] * 86400 : 259200;
          break;
        case "messages_ttl":
          flespiRequestData[field] =
            req.body[field] !== undefined ? req.body[field] * 86400 : 259200;
          break;
        case "configuration":
          flespiRequestData[field] = {
            ident: req.body.uniqueId || undefined,
          };
          break;
        case "metadata":
          const metadataFields = [
            "speedLimit",
            "fuelDropThreshold",
            "fuelIncreaseThreshold",
            "report.ignoreOdometer",
            "web.reportColor",
            "devicePassword",
            "deviceImage",
            "processing.copyAttributes",
            "decoder.timezone",
            "category",
            "groupId",
            "model",
            "contact",
            "phone",
            "expirationTime",
            "disabled",
            "attributes",
          ];
          flespiRequestData[field] = {};

          metadataFields.forEach((metadataField) => {
            if (req.body[metadataField] !== undefined) {
              flespiRequestData[field][metadataField] = req.body[metadataField];
            }
          });
          break;
        default:
          if (req.body[field] !== undefined) {
            flespiRequestData[field] = req.body[field];
          }
      }
    });

    const [flespiResponse, traccarResponse] = await Promise.all([
      
      axios.put(
        `${flespiApiUrl}/devices/${device.flespiId}?fields=id%2Cname%2Cconfiguration%2Cmetadata%2Cdevice_type_id%2Cdevice_type_name%2Cprotocol_id%2Cprotocol_name`,
        flespiRequestData,
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: flespiToken,
          },
        }
      ),
      axios.put(
        `${traccarApiUrl}/devices/${device.traccarId}`,
        traccarRequestData,
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${traccarBearerToken}`,
          },
        }
      ),
    ]);

    const responses = {
      traccar: traccarResponse.data,
      password: req.body.password,
      flespi: {
        ...flespiResponse.data.result[0],
        media_ttl: req.body.media_ttl !== undefined ? req.body.media_ttl : 3,
        messages_ttl:
          req.body.messages_ttl !== undefined ? req.body.messages_ttl : 3,
      },
    };

    // console.log("RESponse : ", responses);

    const update = await updateDeviceById(device.id, responses);

    update.attributes = JSON.parse(update.attributes);

    res.status(200).json({
      status: true,
      message: "device update successfully",
      data: update,
    });
  } catch (error) {
    try {
      if (error?.response?.data?.errors) {
        console.error(
          "Error in creating device response data errors:",
          error.response.data.errors
        );

        let errorMessage = "Flespi Error: ";

        const flespiErrors = error.response.data.errors;

        if (flespiErrors && flespiErrors.length > 0) {
          errorMessage += flespiErrors
            .map((err) => `${err.code}: ${err.reason}`)
            .join(", ");
          console.error(errorMessage);
        } else {
          console.error("Unexpected Flespi error format.");
        }

        res.status(error.response.status).json({
          status: false,
          errorMessage: errorMessage,
        });
      } else {
        console.error("Error in creating device:", error);

        let errorMessage = "Internal Server Error";

        if (error.response) {
          const traccarErrorString = error.response.data.toString();

          if (traccarErrorString.includes("Data truncation")) {
            errorMessage =
              "Traccar Error: Data truncation issue. Check date format and value.";
          } else if (traccarErrorString.includes("Duplicate entry")) {
            errorMessage =
              "Traccar Error: Duplicate entry issue. Choose a different unique identifier.";
          } else {
            errorMessage =
              "Traccar Error: Bad Request. Please check your request data.";
          }
        } else {
          console.log("Unexpected error occurred. Please try again.");
        }

        res.status(error.response ? error.response.status : 500).json({
          status: false,
          errorMessage: errorMessage,
        });
      }
    } catch (catchError) {
      console.error("Unexpected error during error handling:", catchError);
      res.status(500).json({
        status: false,
        errorMessage: "Unexpected error occurred.",
      });
    }
  }
});

router.delete("/new-devices/:id", async (req, res) => {
  try {
    const deviceId = req.params.id;
    // console.log(`Deleting device with ID: ${deviceId}`);

    const device = await getDeviceById(deviceId);
    if (!device) {
      return res.status(404).json({
        status: false,
        error: "Device not found",
      });
    }
    console.log("Retrieved device:", device);

    const [
      traccarResponse
      // , flespiResponse
    ] = await Promise.allSettled([
      axios.delete(`${traccarApiUrl}/devices/${device.traccarId}`, {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }),
      // axios.delete(`${flespiApiUrl}/devices/${device.flespiId}`, {
      //   headers: {
      //     Authorization: flespiToken,
      //   },
      // }),
    ]);

    const failedRequests = [
      traccarResponse
      // , flespiResponse
    ].filter(
      (result) => result.status === "rejected"
    );

    if (failedRequests.length > 0) {
      const errorDetails = failedRequests.map(
        (result) => result.reason.message
      );
      console.error("Failed Requests:", errorDetails);
      return res.status(500).json({
        status: false,
        error: "Failed to delete from external services",
        details: errorDetails,
      });
    }

    const update = await softDeleteDeviceById(deviceId);
    console.log("Device deleted successfully.");

    res.status(200).json({
      status: true,
      message: "Device Deleted successfully.",
      // device
    });
  } catch (error) {
    console.error("Error:", error);

    if (error.response && error.response.data && error.response.data.errors) {
      const serverErrors = error.response.data.errors;
      console.error("Server Errors:", serverErrors);
      res.status(400).json({
        status: false,
        error: "Bad Request",
        details: serverErrors,
      });
    } else {
      res.status(500).json({
        status: false,
        error: "Internal Server Error",
      });
    }
  }
});

router.post("/delete-multiple-devices", async (req, res) => {
  try {
    const deviceIds = req.body;

    // Perform deletion for each device ID
    const deletePromises = deviceIds.map(async (deviceId) => {
      // You may need to adjust this based on your actual API and response structure
      const traccarResponse = await axios.delete(
        `${traccarApiUrl}/devices/${deviceId}`,
        {
          headers: {
            Authorization: `Bearer ${traccarBearerToken}`,
          },
        }
      );
      // Check the response and handle success/failure if needed
      if (traccarResponse.status !== 204) {
        throw new Error(`Failed to delete device with ID ${deviceId}`);
      }
      return deviceId;
    });

    // Wait for all deletion promises to settle
    const results = await Promise.allSettled(deletePromises);

    // Check the results and handle success/failure if needed
    const successfulDeletions = results
      .filter((result) => result.status === "fulfilled")
      .map((result) => result.value);

    if (successfulDeletions.length > 0) {
      res.status(200).json({
        status: true,
        message: `${successfulDeletions.length} ${
          successfulDeletions.length > 1 ? "devices" : "device"
        } deleted successfully.`,
      });
    } else {
      res.status(500).json({
        status: false,
        error: "Failed to delete any devices",
      });
    }
  } catch (error) {
    console.error(error);

    if (error.response && error.response.data && error.response.data.errors) {
      const serverErrors = error.response.data.errors;
      console.error("Server Errors:", serverErrors);
      res.status(400).json({
        status: false,
        error: "Bad Request",
        details: serverErrors,
      });
    } else {
      res.status(500).json({
        status: false,
        error: "Internal Server Error",
      });
    }
  }
});

router.post("/new-devices/find", async (req, res) => {
  const { flespiId, uniqueId, password } = req.body;

  if ((!flespiId && !uniqueId) || !password) {
    return res.status(400).json({ status: false, error: 'Both UniqueID and password are required' });
  }

  try {
    let device;
    if (flespiId) {
      device = await findDeviceByFlespiIdAndPassword(flespiId, password);
    } else if (uniqueId) {
      device = await findDeviceByUniqueIdAndPassword(uniqueId, password);
    }

    if (device) {
      const token = jwt.sign({ id: device.id }, secretKey, { expiresIn: '1h' });
      res.status(200).json({ status: true, data: device, token });
    } else {
      res.status(404).json({ status: false, error: 'Invalid Credentials' });
    }
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: false, error: 'An error occurred while searching for the device' });
  }
});

router.get("/new-devices/:id", async (req, res) => {
  try {
    const deviceId = req.params.id;
    const device = await getDeviceById(deviceId);

    if (!device) {
      res.status(404).json({
        status: false,
        error: "Device not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: device,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});


// ______________________________________________________________________________________________________________________

router.post("/new-users", async (req, res) => {
  try {
    // const requestData = req.body;
    const {
      userId,
      expirationTime,
      isSuperAdmin,
      flespiUserToken,
      ...restFields
    } = req.body;
    // const isSuperAdmin = typeof requestData.isSuperAdmin === "boolean" ? requestData.isSuperAdmin : undefined;

    // If isSuperAdmin is not provided or not a boolean, return an error
    if (isSuperAdmin === undefined) {
      return res.status(400).json({
        status: false,
        error: "isSuperAdmin field is required and must be a boolean.",
      });
    }

    // If isSuperAdmin is false, check for the presence of flespiUserToken
    if (!isSuperAdmin && !flespiUserToken) {
      return res.status(400).json({
        status: false,
        error: "flespiUserToken is required for non-superAdmin users.",
      });
    }

    // Flespi API call
    const flespiResponse = await axios.post(
      "https://flespi.io/platform/subaccounts?fields=id%2Cname%2Cemail%2Cmetadata",
      [{ name: req.body.name, metadata: restFields }],
      {
        headers: {
          Authorization: isSuperAdmin ? flespiToken : flespiUserToken,
        },
      }
    );

    // Traccar API call
    const traccarResponse = await axios.post(
      `http://${process.env.TraccarPort}/api/users`,
      {
        ...restFields,
      },
      {
        headers: { Authorization: `Bearer ${process.env.TraccarToken}` },
      }
    );

    let responses = {
      userId: req.body.userId,
      traccar: traccarResponse.data,
      flespi: flespiResponse.data.result[0],
    };

    const insertID = await createUser(responses);

    responses.id = insertID;

    res.status(200).json({
      status: true,
      message: "User created successfully",
      data: responses,
    });
  } catch (error) {
    console.error(
      "Error in creating group:",
      error.response ? error.response.data : error.message
    );

    let errorMessage = "Internal Server Error";
    if (error.response?.data?.includes("Duplicate entry ")) {
      errorMessage = `${req.body.email} already exist`;
    } else if (error.response && error.response.status === 401) {
      errorMessage = "Unauthorized";
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.get("/new-users", async (req, res) => {
  try {
    const users = await getAllUsers();

    if (!users) {
      res.status(404).json({
        status: false,
        error: "Users not found",
      });
      return;
    }


    const transformedUsers = users.map(user => {
      return {
        ...user,
        time: formatTimeISO(user.created_at),
        administrator: user.administrator === 1,
      };
    });

    res.status(200).json({
      status: true,
      data: transformedUsers,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-users/:id", async (req, res) => {
  try {
    const userId = req.params.id;
    const user = await getUserById(userId);

    if (!user) {
      res.status(404).json({
        status: false,
        error: "User not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: user,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-users/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const users = await getUsersByUserId(userId);

    const transformedUsers = users.map(user => {
      return {
        ...user,
        time: formatTimeISO(user.created_at),
        administrator: user.administrator === 1,
      };
    });

    res.status(200).json({
      status: true,
      data: transformedUsers,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/new-users/:id", async (req, res) => {
  const id = req.params.id;
  const user = await getUserById(id);
  if (!user) {
    return res.status(404).json({
      status: false,
      error: "User not found",
    });
  }

  try {
    const { id, traccarId, flespiId, userId, name, created_at, ...restFields } =
      req.body;

    const requestData = {
      id: traccarId,
      name,
      ...restFields,
    };

    const flespiRequest = {
      name,
      metadata: restFields,
    };

    const [traccarResponse, flespiResponse] = await Promise.all([
      axios.put(
        `http://${process.env.TraccarPort}/api/users/${userId}`,
        requestData,
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${traccarBearerToken}`,
          },
        }
      ),
      axios.put(
        `https://flespi.io/platform/subaccounts/${flespiId}?fields=id%2Cname%2Cemail%2Cmetadata`,
        flespiRequest,
        {
          headers: {
            "Content-Type": "application/json",
            Authorization: flespiToken,
          },
        }
      ),
    ]);

    const responses = {
      id,
      traccar: traccarResponse.data,
      flespi: flespiResponse.data.result[0],
    };

    const update = await updateUserById(responses.id, responses);

    res.status(200).json({
      status: true,
      message: "User created successfully",
      data: update,
    });
  } catch (error) {
    console.error(
      "Error in creating group:",
      error.response ? error.response.data : error.message
    );

    let errorMessage = "Internal Server Error";
    if (
      typeof error.response?.data === "string" &&
      error.response?.data.includes("Duplicate entry ")
    ) {
      errorMessage = `${req.body.email} already exists`;
    } else if (error.response && error.response.status === 401) {
      errorMessage = "Unauthorized";
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.delete("/new-users/:id", async (req, res) => {
  try {
    const userId = req.params.id;
    // console.log(`Deleting user with ID: ${userId}`);

    const user = await getUserById(userId);
    if (!user) {
      return res.status(404).json({
        status: false,
        error: "User not found",
      });
    }
    // console.log("Retrieved user:", user);

    const devices = await getDevicesByUserId(user.traccarId);
    // console.log("Retrieved devices:", devices);

    let deletedDevices = 0;
    let notDeletedDevices = 0;

    const deleteDevice = async (deviceId) => {
      try {
        const apiUrl = `${baseURL}/new-devices/${deviceId}`;
        const response = await axios.delete(apiUrl, { timeout: 15000 });

        if (response.status === 200) {
          // console.log(`Device ${deviceId} deleted successfully!`);
          deletedDevices++;
        } else {
          // console.log(`Device ${deviceId} not deleted!`);
          notDeletedDevices++;
        }
      } catch (error) {
        console.error(`Error deleting device ${deviceId}:`, error);
        notDeletedDevices++;
      }
    };

    await Promise.all(
      devices.map(async (device) => {
        await deleteDevice(device.id);
      })
    );

    const [traccarResponse, flespiResponse] = await Promise.allSettled([
      axios.delete(`${traccarApiUrl}/users/${user.traccarId}`, {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }),
      axios.delete(`https://flespi.io/platform/subaccounts/${user.flespiId}`, {
        headers: {
          Authorization: flespiToken,
        },
      }),
    ]);

    // Handle failed requests
    const failedRequests = [traccarResponse, flespiResponse].filter(
      (result) => result.status === "rejected"
    );

    if (failedRequests.length > 0) {
      const errorDetails = failedRequests.map(
        (result) => result.reason.message
      );
      console.error("Failed Requests:", errorDetails);
      return res.status(500).json({
        status: false,
        error: "Failed to delete from external services",
        details: errorDetails,
      });
    }

    // Soft delete user
    const update = await softDeleteUserById(userId);

    res.status(200).json({
      status: true,
      message: `User ${user.name} and associated devices deleted successfully.`,
      deletedDevices,
      notDeletedDevices,
    });
  } catch (error) {
    console.error("Error:", error);

    if (error.response && error.response.data && error.response.data.errors) {
      const serverErrors = error.response.data.errors;
      console.error("Server Errors:", serverErrors);
      res.status(400).json({
        status: false,
        error: "Bad Request",
        details: serverErrors,
      });
    } else {
      res.status(500).json({
        status: false,
        error: "Internal Server Error",
      });
    }
  }
});

router.post("/delete-multiple-users", async (req, res) => {
  try {
    const userIds = req.body;

    // Perform deletion for each device ID
    const deletePromises = userIds.map(async (userId) => {
      // You may need to adjust this based on your actual API and response structure
      const traccarResponse = await axios.delete(
        `${traccarApiUrl}/users/${userId}`,
        {
          headers: {
            Authorization: `Bearer ${traccarBearerToken}`,
          },
        }
      );
      // Check the response and handle success/failure if needed
      if (traccarResponse.status !== 204) {
        throw new Error(`Failed to delete user with ID ${userId}`);
      }
      return userId;
    });

    // Wait for all deletion promises to settle
    const results = await Promise.allSettled(deletePromises);

    // Check the results and handle success/failure if needed
    const successfulDeletions = results
      .filter((result) => result.status === "fulfilled")
      .map((result) => result.value);

    if (successfulDeletions.length > 0) {
      res.status(200).json({
        status: true,
        message: `${successfulDeletions.length} ${
          successfulDeletions.length > 1 ? "users" : "user"
        } deleted successfully.`,
      });
    } else {
      res.status(500).json({
        status: false,
        error: "Failed to delete any users",
      });
    }
  } catch (error) {
    console.error(error);

    if (error.response && error.response.data && error.response.data.errors) {
      const serverErrors = error.response.data.errors;
      console.error("Server Errors:", serverErrors);
      res.status(400).json({
        status: false,
        error: "Bad Request",
        details: serverErrors,
      });
    } else {
      res.status(500).json({
        status: false,
        error: "Internal Server Error",
      });
    }
  }
});

// ______________________________________________________________________________________________________________________

router.post("/existing-user", async (req, res) => {
  try {
    const { traccarId, flespiId, userId } = req.body;

    // Traccar API call
    const traccarResponse = await axios.get(
      `http://${process.env.TraccarPort}/api/users/${traccarId}`,
      // {
      //   ...restFields,
      // },
      {
        headers: { Authorization: `Bearer ${process.env.TraccarToken}` },
      }
    );

    // Flespi API call
    // const flespiResponse = await axios.get(
    //   `https://flespi.io/platform/subaccounts/${flespiId}?fields=id%2Cname%2Cemail%2Cmetadata`,
    //   // [{ name: req.body.name, metadata: restFields }],
    //   {
    //     headers: { Authorization: `FlespiToken ${process.env.FlespiToken}` },
    //   }
    // );

    let responses = {
      userId: userId,
      traccar: traccarResponse.data,
      // flespi: flespiResponse.data.result[0],
    };

    // const insertID = await createUser(responses);

    // responses.id = insertID;

    res.status(200).json({
      status: true,
      message: "User created successfully",
      data: responses,
    });
  } catch (error) {
    console.error(
      "Error in creating group:",
      error.response ? error.response.data : error.message
    );

    let errorMessage = "Internal Server Error";
    if (error.response?.data?.includes("Duplicate entry ")) {
      errorMessage = `${req.body.email} already exist`;
    } else if (error.response && error.response.status === 401) {
      errorMessage = "Unauthorized";
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.post("/uft", async (req, res) => {
  const { cid, expire, access_type, userId, email } = req.body;
  const missingFields = ["cid", "expire", "userId", "email"].filter(
    (field) => !req.body[field]
  );

  if (missingFields.length > 0) {
    return res.status(400).json({
      status: false,
      error: `Required fields are missing: ${missingFields.join(", ")}`,
    });
  }

  try {
    const apiUrl = "https://flespi.io/platform/tokens";
    const headers = {
      Authorization: `FlespiToken ${process.env.FlespiToken}`,
      "x-flespi-cid": cid,
    };
    const requestBody = [
      {
        expire,
        access: {
          type: access_type || 0,
        },
      },
    ];

    const response = await axios.post(apiUrl, requestBody, { headers });

    if (response.data.result && response.data.result.length > 0) {
      const insertedData = response.data.result[0];

      const insertedId = await createUFT({
        ...insertedData,
        userId,
        email,
      });

      const responseData = { ...insertedData, userId, email, insertedId };

      res.status(response.status).json({ status: true, data: responseData });
    } else {
      res.status(500).json({
        status: false,
        error: "Failed to insert data into the database",
      });
    }
  } catch (error) {
    if (error.response && error.response.status === 403) {
      const errorData = error.response.data.errors[0];
      res.status(403).json({ status: false, error: errorData });
    } else {
      console.error(error);
      res.status(500).json({ status: false, error: "Internal server error" });
    }
  }
});

router.post("/uft2", async (req, res) => {
  const { cid, access_type, userId, email, token } = req.body;
  const missingFields = ["cid", "userId", "email", "token"].filter(
    (field) => !req.body[field]
  );

  if (missingFields.length > 0) {
    return res.status(400).json({
      status: false,
      error: `Required fields are missing: ${missingFields.join(", ")}`,
    });
  }

  try {
    if (cid && userId && email && token) {
      const insertedId = await createUFT2({
        cid,
        access_type,
        userId,
        email,
        key: token,
      });

      const responseData = { insertedId };

      res.status(200).json({ status: true, data: responseData });
    } else {
      res.status(500).json({
        status: false,
        error: "Failed to insert data into the database",
      });
    }
  } catch (error) {
    if (error.response && error.response.status === 403) {
      const errorData = error.response.data.errors[0];
      res.status(403).json({ status: false, error: errorData });
    } else {
      console.error(error);
      res.status(500).json({ status: false, error: "Internal server error" });
    }
  }
});

router.get("/uft/:id", async (req, res) => {
  try {
    const userId = req.params.id;
    const token = await getTokenByUserId(userId);

    if (!token) {
      res.status(404).json({
        status: false,
        error: "token not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: token,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

// ______________________________________________________________________________________________________________________

router.post("/snapshots-devices", async (req, res) => {
  try {
    const data = req.body;
    if (!data || !Array.isArray(data)) {
      return res.status(400).json({ error: "Invalid request payload" });
    }

    await addSnapshotsDevices(data);

    res
      .status(200)
      .json({ status: true, message: "Devices stored successfully" });
  } catch (err) {
    console.error("Error processing request:", err);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
});

router.get("/snapshots-devices", async (req, res) => {
  try {
    const allIdsAndNames = await getAllIdsAndNames();

    res.status(200).json({ status: true, data: allIdsAndNames });
  } catch (err) {
    console.error("Error processing request:", err);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
});

router.post("/snapshots-devices-remove", async (req, res) => {
  try {
    const data = req.body;
    if (!Array.isArray(data) || data.length === 0) {
      return res
        .status(400)
        .json({ status: false, error: "Invalid or empty array of deviceIds" });
    }
    await removeDevicesByIds(data);
    res
      .status(200)
      .json({ status: true, message: "Devices removed successfully" });
  } catch (err) {
    console.error("Error processing request:", err);
    res.status(500).json({ status: false, error: "Internal server error" });
  }
});

router.get("/snapshots", async (req, res) => {
  try {
    const snapshots = await getAllSnapshots();

    if (!snapshots) {
      res.status(404).json({
        status: false,
        error: "Users not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: snapshots,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-snapshots", async (req, res) => {
  try {
    const snapshots = await getAllSnapshots2();

    if (!snapshots) {
      res.status(404).json({
        status: false,
        error: "Users not found",
      });
      return;
    }

    const transformedSnapshots = snapshots.map(snapshot => {
      return {
        ...snapshot,
        startTime: formatTimestamp(snapshot.startTimestamp),
      };
    });

    res.status(200).json({
      status: true,
      data: transformedSnapshots,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/snapshots/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const snapshots = await getSnapshotsByUserId(userId);

    res.status(200).json({
      status: true,
      data: snapshots,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/snapshots/by-flespi-ids", async (req, res) => {
  try {
    const { flespiIds } = req.query;

    if (!flespiIds) {
      res.status(400).json({
        status: false,
        error: "Invalid or missing 'flespiIds' parameter",
      });
      return;
    }

    const flespiIdsArray = flespiIds.split(",").map(Number);
    const snapshots = await getSnapshotsByFlespiIds(flespiIdsArray);

    res.status(200).json({
      status: true,
      data: snapshots,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-snapshots/by-flespi-ids", async (req, res) => {
  try {
    const { flespiIds } = req.query;

    if (!flespiIds) {
      res.status(400).json({
        status: false,
        error: "Invalid or missing 'flespiIds' parameter",
      });
      return;
    }

    const flespiIdsArray = flespiIds.split(",").map(Number);
    const snapshots = await getSnapshots2ByFlespiIds(flespiIdsArray);

    const transformedSnapshots = snapshots.map(snapshot => {
      return {
        ...snapshot,
        startTime: formatTimestamp(snapshot.startTimestamp),
      };
    });

    res.status(200).json({
      status: true,
      data: transformedSnapshots,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

// router.post("/snapshots", async (req, res) => {
//   try {
//     const AvaibleSnapshots = await getAllSnapshots();

//     const getFlespiSnapshots = async () => {
//       try {
//         const response = await fetch(
//           `https://flespi.io/gw/devices/all/snapshots`,
//           {
//             headers: {
//               Authorization: `FlespiToken ${process.env.FlespiToken}`,
//             },
//           }
//         );
//         // console.log("snpashots response", response);
//         if (response.status === 200) {
//           const data = await response.json();
//           console.log("Successfully fetched all flespi snapshots");
//           return data.result;
//         }
//       } catch (error) {
//         console.error("Error fetching all snapshots");
//         return null;
//       }
//     };

//     const getFelspiDevices = async () => {
//       try {
//         const response = await fetch(
//           `https://flespi.io/gw/devices/all?fields=id,name`,
//           {
//             headers: {
//               Authorization: `FlespiToken ${process.env.FlespiToken}`,
//             },
//           }
//         );

//         // console.log("Devices response", response);
//         if (response.status === 200) {
//           const data = await response.json();
//           console.log("Successfully fetched all db snapshots");
//           return data.result;
//         }
//       } catch (error) {
//         console.error("Error fetching all snapshots");
//         return null;
//       }
//     };

//     const getYesterdayEndTimestampGMT = () => {
//       const yesterday = new Date();
//       yesterday.setUTCHours(0, 0, 0, 0);
//       yesterday.setUTCDate(yesterday.getUTCDate() - 1);
//       yesterday.setUTCHours(23, 59, 59, 999);
//       return Math.floor(yesterday.getTime() / 1000);
//     };

//     const flesiSnapshots = await getFlespiSnapshots();
//     const flespiDevices = await getFelspiDevices();

//     if (!flesiSnapshots || !flespiDevices) {
//       console.error("Failed to fetch Flespi snapshots or devices");
//       res.status(500).json({
//         status: false,
//         message: "Failed to fetch Flespi snapshots or devices",
//       });
//       return;
//     }

//     const filterOutdatedAndUnavailableSnapshots = (
//       snapshots,
//       availableSnapshots,
//       devices
//     ) => {
//       const YesterdayEndTimestampGMT = getYesterdayEndTimestampGMT();

//       return snapshots.flatMap((snapshot) => {
//         const matchingAvailableSnapshot = availableSnapshots.find(
//           (available) =>
//             snapshot.id === available.flespiId &&
//             snapshot.snapshots.includes(available.snapshot)
//         );

//         if (matchingAvailableSnapshot) {
//           snapshot.snapshots = snapshot.snapshots.filter(
//             (timestamp) => timestamp !== matchingAvailableSnapshot.snapshot
//           );
//         }

//         snapshot.snapshots = snapshot.snapshots.filter(
//           (timestamp) => timestamp < YesterdayEndTimestampGMT
//         );

//         const device = devices.find((d) => d.id === snapshot.id);
//         const deviceName = device ? device.name : undefined;

//         return snapshot.snapshots.map((timestamp) => ({
//           id: snapshot.id,
//           snapshot: timestamp,
//           name: deviceName,
//         }));
//       });
//     };

//     const filteredSnapshots = filterOutdatedAndUnavailableSnapshots(
//       flesiSnapshots,
//       AvaibleSnapshots,
//       flespiDevices
//     );

//     res.status(200).json({
//       status: true,
//       message: "Process working in backgroud...",
//     });

//     for (const snapshotData of filteredSnapshots) {
//       console.log(
//         `Processing device ID: ${snapshotData.id}, device Name: ${snapshotData.name} Snapshot: ${snapshotData.snapshot}`
//       );

//       try {
//         const flespiResponse = await axios.get(
//           `https://flespi.io/gw/devices/${snapshotData.id}/snapshots/${snapshotData.snapshot}`,
//           {
//             headers: {
//               Authorization: `FlespiToken ${process.env.FlespiToken}`,
//             },
//             responseType: "stream",
//           }
//         );

//         if (flespiResponse.headers["content-length"] === "0") {
//           console.log("Snapshot is empty");
//           continue;
//         }

//         const decompressedStream = flespiResponse.data.pipe(createGunzip());

//         const zip = new JSZip();

//         // Adding the decompressed content to the zip file
//         zip.file(
//           `${snapshotData.id}-${snapshotData.snapshot}.txt`,
//           decompressedStream
//         );

//         // Generating the zip archive
//         const zipBlob = await zip.generateAsync({ type: "nodebuffer" });

//         const s3Client = new S3({
//           region: "US-central",
//           credentials: {
//             accessKeyId: process.env.CONTABO_ACCESS_KEY_ID,
//             secretAccessKey: process.env.CONTABO_SECRET_ACCESS_KEY,
//           },
//           endpoint: "https://usc1.contabostorage.com",
//           forcePathStyle: true,
//         });

//         const uploadResult = await s3Client.putObject({
//           Bucket: "test",
//           Key: `${snapshotData.name}(${snapshotData.id})-${snapshotData.snapshot}.zip`,
//           Body: zipBlob,
//           ACL: "public-read",
//         });

//         console.log("Upload Result :", uploadResult);

//         const insertData = {
//           flespiId: snapshotData.id,
//           snapshot: snapshotData.snapshot,
//           deviceName: snapshotData.name,
//           s3Path: `https://usc1.contabostorage.com/e3ea2b21c1414932b7696559a9f1db58:test/${snapshotData.name}(${snapshotData.id})-${snapshotData.snapshot}.zip`,
//         };

//         const insertID = await createSnapshot(insertData);

//         insertData.id = insertID;

//         console.log("Data uploaded to S3 successfully", insertData);
//       } catch (error) {
//         if (error.response && error.response.status === 403) {
//           console.error("Forbidden", error.message);
//         } else {
//           console.error("API Response error", error);
//         }
//       }
//     }
//   } catch (error) {
//     console.error("Error processing snapshots:", error);
//     res.status(500).json({
//       status: false,
//       error: "Internal Server Error",
//     });
//   }
// });

router.post("/snapshots", async (req, res) => {
  try {
    const devicesForSnapshots = await getAllIdsAndNames();
    const getFlespiSnapshots = async () => {
      try {
        const response = await fetch(
          `https://flespi.io/gw/devices/all/snapshots`,
          {
            headers: {
              Authorization: `FlespiToken ${process.env.FlespiToken}`,
            },
          }
        );
        if (response.status === 200) {
          const data = await response.json();
          console.log("Successfully fetched all flespi snapshots");
          return data.result;
        }
      } catch (error) {
        console.error("Error fetching all snapshots");
        return null;
      }
    };
    const flesiSnapshots = await getFlespiSnapshots();

    const getYesterdayTimestampsGMT = () => {
      const now = new Date();
      const yesterdayStart = new Date(now);
      yesterdayStart.setUTCHours(0, 0, 0, 0);
      yesterdayStart.setUTCDate(now.getUTCDate() - 1);

      const yesterdayEnd = new Date(now);
      yesterdayEnd.setUTCHours(23, 59, 59, 999);
      yesterdayEnd.setUTCDate(now.getUTCDate() - 1);

      return {
        startTimestamp: Math.floor(yesterdayStart.getTime() / 1000),
        endTimestamp: Math.floor(yesterdayEnd.getTime() / 1000),
      };
    };

    const { startTimestamp, endTimestamp } = getYesterdayTimestampsGMT();

    const filterLastSnapshotByDate = (
      devicesForSnapshots,
      flesiSnapshots,
      startTimestamp,
      endTimestamp
    ) => {
      return devicesForSnapshots.flatMap((device) => {
        const snapshotsData = flesiSnapshots.find((snap) => snap.id === device.id);
    
        if (snapshotsData) {
          const filteredSnapshots = snapshotsData.snapshots.filter(
            (snapshot) => snapshot >= startTimestamp && snapshot <= endTimestamp
          );
    
          const lastSnapshot =
            filteredSnapshots.length > 0
              ? filteredSnapshots[filteredSnapshots.length - 1]
              : null;
    
          if (lastSnapshot !== null) {
            return {
              id: device.id,
              name: device.name,
              snapshot: lastSnapshot,
            };
          }
        }
    
        return [];
      });
    };
    
    const filteredLastSnapshot = filterLastSnapshotByDate(
      devicesForSnapshots,
      flesiSnapshots,
      startTimestamp,
      endTimestamp
    );

    res.status(200).json({
      status: true,
      message: "Process working in backgroud...",
      // filteredLastSnapshot
    });

    for (const snap of filteredLastSnapshot) {
      console.log("snap", snap)
      console.log(
        `Processing device ID: ${snap.id}, device Name: ${snap.name} Snapshot: ${snap.snapshot}`
      );

      try {
        const flespiResponse = await axios.get(
          `https://flespi.io/gw/devices/${snap.id}/snapshots/${snap.snapshot}`,
          {
            headers: {
              Authorization: `FlespiToken ${process.env.FlespiToken}`,
            },
            responseType: "stream",
          }
        );

        if (flespiResponse.headers["content-length"] === "0") {
          console.log("Snapshot is empty");
          continue;
        }

        const decompressedStream = flespiResponse.data.pipe(createGunzip());

        const zip = new JSZip();

        // Adding the decompressed content to the zip file
        zip.file(`${snap.id}-${snap.snapshot}.txt`, decompressedStream);

        // Generating the zip archive
        const zipBlob = await zip.generateAsync({ type: "nodebuffer" });

        const s3Client = new S3({
          region: "US-central",
          credentials: {
            accessKeyId: process.env.CONTABO_ACCESS_KEY_ID,
            secretAccessKey: process.env.CONTABO_SECRET_ACCESS_KEY,
          },
          endpoint: "https://usc1.contabostorage.com",
          forcePathStyle: true,
        });

        const uploadResult = await s3Client.putObject({
          Bucket: "test",
          Key: `${snap.name}(${snap.id}).zip`,
          Body: zipBlob,
          ACL: "public-read",
        });

        console.log("Upload Result :", uploadResult);

        const insertData = {
          flespiId: snap.id,
          snapshot: snap.snapshot,
          deviceName: snap.name,
          s3Path: `https://usc1.contabostorage.com/e3ea2b21c1414932b7696559a9f1db58:test/${snap.name}(${snap.id}).zip`,
        };

        const insertID = await createSnapshot(insertData);

        insertData.id = insertID;

        console.log("Data uploaded to S3 successfully", insertData);
      } catch (error) {
        if (error.response && error.response.status === 403) {
          console.error("Forbidden", error.message);
        } else {
          console.error("API Response error", error);
        }
      }
    }

  } catch (error) {
    console.error("Error processing snapshots:", error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.post("/new-snapshots", async (req, res) => {
  try {
      const devicesForSnapshots = await getAllIdsAndNames();
      const contaboEndpoint = process.env.CONTABO_ENDPOINT;
      const conatboBucket = process.env.CONTABO_BUCKET_NAME;
      const contaboBucketAddress = process.env.CONTABO_BUCKET_ADDRESS;
      const getYesterdayTimestampsGMT = () => {
          const now = new Date();
          const yesterdayStart = new Date(now);
          yesterdayStart.setUTCHours(0, 0, 0, 0);
          yesterdayStart.setUTCDate(now.getUTCDate() - 1);

          const yesterdayEnd = new Date(now);
          yesterdayEnd.setUTCHours(23, 59, 59, 999);
          yesterdayEnd.setUTCDate(now.getUTCDate() - 1);

          return {
              startTimestamp: Math.floor(yesterdayStart.getTime() / 1000),
              endTimestamp: Math.floor(yesterdayEnd.getTime() / 1000),
          };
      };

      const { startTimestamp, endTimestamp } = getYesterdayTimestampsGMT();

      res.status(200).json({
        status: true,
        message: "Process working in backgroud...",
      });

      for (const device of devicesForSnapshots) {
          console.log(
              `Processing device ID: ${device.id}, device Name: ${device.name} startTimestamp: ${startTimestamp}  endTimestamp: ${endTimestamp}`
          );

          try {
              const flespiResponse = await axios.get(
            `https://flespi.io/gw/devices/${device.id}/messages?data=%7B%22fields%22%3A%22battery.voltage%2Ccan.engine.ignition.status%2Ccan.fuel.consumed%2Ccan.vehicle.mileage%2Cdevice.name%2Cdin%2Cdout%2Cevent.enum%2Cexternal.powersource.voltage%2Cfuel.flow.meter.fuel.consumed%2Cgsm.cellid%2Cgsm.signal.dbm%2Cident%2Cmovement.status%2Cposition.altitude%2Cposition.direction%2Cposition.hdop%2Cposition.latitude%2Cposition.longitude%2Cposition.satellites%2Cposition.speed%2Crfid.code%2Ctimestamp%2Cvehicle.mileage%22%2C%22from%22%3A${startTimestamp}%2C%22to%22%3A${endTimestamp}%2C%22method%22%3A%22average%22%7D`,
            {
              headers: {
                Authorization: `FlespiToken ${process.env.FlespiToken}`,
              },
            }
          );

              if (flespiResponse.headers["content-length"] === "0") {
                  console.log("Snapshot is empty");
                  continue;
              }

              const json2csvParser = new Parser();
              const csvData = json2csvParser.parse(flespiResponse.data.result);

              const csvBuffer = Buffer.from(csvData, 'utf-8');
  
              const csvSizeInBytes = csvBuffer.length;

              const zip = new JSZip();
                zip.file(`${device.name} - ${formatTimestamp(startTimestamp)}.csv`, csvData);

                const zipData = await zip.generateAsync({ type: 'nodebuffer' });

              const s3Client = new S3({
                  region: "US-central",
                  credentials: {
                      accessKeyId: process.env.CONTABO_ACCESS_KEY_ID,
                      secretAccessKey: process.env.CONTABO_SECRET_ACCESS_KEY,
                  },
                  endpoint: contaboEndpoint,
                  forcePathStyle: true,
              });

              const uploadResult = await s3Client.putObject({
                Bucket: conatboBucket,
                Key: `${device.name} - ${formatTimestamp(startTimestamp)}.zip`,
                ACL: "public-read",
                Body: zipData,
            });
            
            const filePath = `${contaboEndpoint}/${contaboBucketAddress}:${conatboBucket}/${device.name} - ${formatTimestamp(startTimestamp)}.zip`;
            
            
            const insertData = {
              deviceId: device.id,
              deviceName: device.name,
              startTimestamp: startTimestamp,
              endTimestamp: endTimestamp,
              contaboEndpoint: contaboEndpoint,
              contaboBucketName: conatboBucket,
              contaboBucketAddress: contaboBucketAddress,
              fileExtension: 'zip/csv',
              size: csvSizeInBytes,
              filePath: filePath,
              uploadResult: uploadResult,
            };
            
            const insertID = await createSnapshot2(insertData);

            insertData.id = insertID;
    
            console.log("Data uploaded to S3 successfully", insertData);

          } catch (error) {
              if (error.response && error.response.status === 403) {
                  console.error("Forbidden", error.message);
              } else {
                  console.error("API Response error", error);
              }
          }
      }

  } catch (error) {
      console.error("Error processing snapshots:", error);
      res.status(500).json({
          status: false,
          error: "Internal Server Error",
      });
  }
});


router.get('/files', async (req, res) => {
  const s3Client = new S3({
    region: "US-central",
    credentials: {
      accessKeyId: process.env.CONTABO_ACCESS_KEY_ID,
      secretAccessKey: process.env.CONTABO_SECRET_ACCESS_KEY,
    },
    endpoint: "https://usc1.contabostorage.com",
    forcePathStyle: true,
  });

  try {
    let files = [];
    let continuationToken = null;

    do {
      const command = new ListObjectsV2Command({
        Bucket: 'test-2',
        ContinuationToken: continuationToken,
      });
      const data = await s3Client.send(command);
      files = files.concat(data.Contents.map(file => file.Key));
      continuationToken = data.NextContinuationToken;
    } while (continuationToken);

    res.json({ files });
  } catch (error) {
    console.error('Error fetching files:', error);
    res.status(500).json({ error: 'Internal Server Error' });
  }
});


router.post('/store-snapshots', async (req, res) => {
  try {
    const data = req.body;

    const promises = data.map(async (file) => {
      await createSnapshot({
        deviceName: file.name,
        s3Path: file.path
      });
    });

    await Promise.all(promises);

    res.status(200).json({ message: 'Data stored successfully' });
  } catch (error) {
    console.error('Error storing data:', error);
    res.status(500).json({ error: 'Internal Server Error' });
  }
});


// ______________________________________________________________________________________________________________________

router.post("/new-geofences", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...restFields } = req.body;

    if (
      userId === undefined ||
      userId === null ||
      isNaN(userId) ||
      typeof userId !== "number"
    ) {
      return res.status(400).json({
        status: false,
        error: "User ID is required.",
      });
    }

    if (isSuperAdmin === undefined) {
      return res.status(400).json({
        status: false,
        error: "isSuperAdmin field is required and must be a boolean.",
      });
    }

    if (!isSuperAdmin && !traccarUserToken) {
      return res.status(400).json({
        status: false,
        error: "Nextop Token is required.",
      });
    }
    const { attributes, calendarId, description, area, name } = restFields;
    const traccarResponse = await axios.post(
      `http://${process.env.TraccarPort}/api/geofences`,
      {
        attributes,
        calendarId,
        description,
        area,
        name,
      },
      {
        headers: {
          Authorization: isSuperAdmin
            ? `Bearer ${traccarBearerToken}`
            : `Bearer ${traccarUserToken}`,
        },
      }
    );

    let responses = {
      userId: userId,
      traccar: traccarResponse.data,
    };

    const insertID = await createGeofence(responses);

    responses.id = insertID;

    res.status(200).json({
      status: true,
      message: "Geofence Created successfully.",
      data: responses,
    });
  } catch (error) {
    console.error(
      "Error in creating Geofence:",
      error.response ? error.response.data : error.message
    );

    let errorMessage = "Internal Server Error";
    if (error.response?.data?.includes("Duplicate entry ")) {
      errorMessage = `${req.body.email} already exist`;
    } else if (error.response && error.response.status === 401) {
      errorMessage = "Unauthorized";
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.get("/new-geofences", async (req, res) => {
  try {
    const geofences = await getAllGeofences();

    if (!geofences) {
      res.status(404).json({
        status: false,
        error: "Geofences not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: geofences,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-geofences/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getGeofenceById(id);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Geofence not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-geofences/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const result = await getGeofencesByUserId(userId);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Geofences not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/new-geofences/:id", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...Fields } = req.body;
    const id = req.params.id;
    const result = await getGeofenceById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Geofence not found",
      });
    }
    const { attributes, calendarId, description, area, name } = Fields;
    const requestData = {
      id: result.traccarId,
      attributes,
      calendarId,
      description,
      area,
      name,
    };

    const response = await axios.put(
      `http://${process.env.TraccarPort}/api/geofences/${requestData.id}`,
      requestData,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    const update = await updateGeofenceById(id, response.data);

    return res.status(200).json({
      status: true,
      message: "Geofence Update successfully.",
      data: update,
    });
  } catch (error) {
    console.error("Error in updating geofence:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.delete("/new-geofences/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getGeofenceById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Geofence not found",
      });
    }

    const response = await axios.delete(
      `http://${process.env.TraccarPort}/api/geofences/${result.traccarId}`,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    if (response.status === 204) {
      await softDeleteGeofenceById(id);
    }

    return res.status(200).json({
      status: true,
      message: "Geofence Deleted successfully.",
    });
  } catch (error) {
    console.error("Error in updating geofence:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

// ______________________________________________________________________________________________________________________

router.post("/new-calendars", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...restFields } = req.body;

    if (
      userId === undefined ||
      userId === null ||
      isNaN(userId) ||
      typeof userId !== "number"
    ) {
      return res.status(400).json({
        status: false,
        error: "User ID is required.",
      });
    }

    if (isSuperAdmin === undefined) {
      return res.status(400).json({
        status: false,
        error: "isSuperAdmin field is required and must be a boolean.",
      });
    }

    if (!isSuperAdmin && !traccarUserToken) {
      return res.status(400).json({
        status: false,
        error: "Nextop Token is required.",
      });
    }
    const { name, data, attributes } = restFields;
    const traccarResponse = await axios.post(
      `http://${process.env.TraccarPort}/api/calendars`,
      {
        name,
        data,
        attributes,
      },
      {
        headers: {
          Authorization: isSuperAdmin
            ? `Bearer ${traccarBearerToken}`
            : `Bearer ${traccarUserToken}`,
        },
      }
    );

    let responses = {
      userId: userId,
      traccar: traccarResponse.data,
    };

    const insertID = await createCalendar(responses);

    responses.id = insertID;

    res.status(200).json({
      status: true,
      message: "Calendar Created successfully.",
      data: responses,
    });
  } catch (error) {
    console.error(
      "Error in creating Calendar:",
      error.response ? error.response.data : error.message
    );

    let errorMessage = "Internal Server Error";
    if (error.response?.data?.includes("Duplicate entry ")) {
      errorMessage = `${req.body.email} already exist`;
    } else if (error.response && error.response.status === 401) {
      errorMessage = "Unauthorized";
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.get("/new-calendars", async (req, res) => {
  try {
    const calendars = await getAllCalendars();

    if (!calendars) {
      res.status(404).json({
        status: false,
        error: "Calendars not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: calendars,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-calendars/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getCalendarById(id);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Calendar not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-calendars/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const result = await getCalendarsByUserId(userId);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Calendars not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/new-calendars/:id", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...Fields } = req.body;
    const id = req.params.id;
    const result = await getCalendarById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Calendar not found",
      });
    }

    const { name, data, attributes } = Fields;
    const requestData = {
      id: result.traccarId,
      name,
      data,
      attributes,
    };

    const response = await axios.put(
      `http://${process.env.TraccarPort}/api/calendars/${requestData.id}`,
      requestData,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    const update = await updateCalendarById(id, response.data);

    return res.status(200).json({
      status: true,
      message: "Calendar Update successfully.",
      data: update,
    });
  } catch (error) {
    console.error("Error in updating calendar:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.delete("/new-calendars/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getCalendarById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Calendar not found",
      });
    }

    const response = await axios.delete(
      `http://${process.env.TraccarPort}/api/calendars/${result.traccarId}`,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    if (response.status === 204) {
      await softDeleteCalendarById(id);
    }

    return res.status(200).json({
      status: true,
      message: "Calendar Deleted successfully.",
    });
  } catch (error) {
    console.error("Error in updating calendar:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

// ______________________________________________________________________________________________________________________

router.post("/new-attributes", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...restFields } = req.body;

    if (
      userId === undefined ||
      userId === null ||
      isNaN(userId) ||
      typeof userId !== "number"
    ) {
      return res.status(400).json({
        status: false,
        error: "User ID is required.",
      });
    }

    if (isSuperAdmin === undefined) {
      return res.status(400).json({
        status: false,
        error: "isSuperAdmin field is required and must be a boolean.",
      });
    }

    if (!isSuperAdmin && !traccarUserToken) {
      return res.status(400).json({
        status: false,
        error: "Nextop Token is required.",
      });
    }
    const { description, attribute, expression, type } = restFields;
    const traccarResponse = await axios.post(
      `http://${process.env.TraccarPort}/api/attributes/computed`,
      {
        description,
        attribute,
        expression,
        type,
      },
      {
        headers: {
          Authorization: isSuperAdmin
            ? `Bearer ${traccarBearerToken}`
            : `Bearer ${traccarUserToken}`,
        },
      }
    );

    let responses = {
      userId: userId,
      traccar: traccarResponse.data,
    };

    const insertID = await createAttribute(responses);

    responses.id = insertID;

    res.status(200).json({
      status: true,
      message: "Compute Attribute Created successfully.",
      data: responses,
    });
  } catch (error) {
    console.error(
      "Error in creating Compute Attribute:",
      error.response ? error.response.data : error.message
    );

    let errorMessage = "Internal Server Error";
    if (error.response?.data?.includes("Duplicate entry ")) {
      errorMessage = `${req.body.email} already exist`;
    } else if (error.response && error.response.status === 401) {
      errorMessage = "Unauthorized";
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.get("/new-attributes", async (req, res) => {
  try {
    const attributes = await getAllAttributes();

    if (!attributes) {
      res.status(404).json({
        status: false,
        error: "Compute Attributes not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: attributes,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-attributes/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getAttributeById(id);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Compute Attribute not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-attributes/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const result = await getAttributesByUserId(userId);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Compute Attributes not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/new-attributes/:id", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...Fields } = req.body;
    const id = req.params.id;
    const result = await getAttributeById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Compute Attribute not found",
      });
    }
    const { description, attribute, expression, type } = Fields;
    const requestData = {
      id: result.traccarId,
      description,
      attribute,
      expression,
      type,
    };

    const response = await axios.put(
      `http://${process.env.TraccarPort}/api/attributes/computed/${requestData.id}`,
      requestData,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    const update = await updateAttributeById(id, response.data);

    return res.status(200).json({
      status: true,
      message: "Compute Attribute Update successfully.",
      data: update,
    });
  } catch (error) {
    console.error("Error in updating compute attribute:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.delete("/new-attributes/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getAttributeById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Compute Attribute not found",
      });
    }

    const response = await axios.delete(
      `http://${process.env.TraccarPort}/api/attributes/computed/${result.traccarId}`,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    if (response.status === 204) {
      await softDeleteAttributeById(id);
    }

    return res.status(200).json({
      status: true,
      message: "Compute Attribute Deleted successfully.",
    });
  } catch (error) {
    console.error("Error in updating compute attribute:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

// ______________________________________________________________________________________________________________________

router.post("/new-commands", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...restFields } = req.body;

    if (
      userId === undefined ||
      userId === null ||
      isNaN(userId) ||
      typeof userId !== "number"
    ) {
      return res.status(400).json({
        status: false,
        error: "User ID is required.",
      });
    }

    if (isSuperAdmin === undefined) {
      return res.status(400).json({
        status: false,
        error: "isSuperAdmin field is required and must be a boolean.",
      });
    }

    if (!isSuperAdmin && !traccarUserToken) {
      return res.status(400).json({
        status: false,
        error: "Nextop Token is required.",
      });
    }

    const traccarResponse = await axios.post(
      `http://${process.env.TraccarPort}/api/commands`,
      {
        ...restFields,
      },
      {
        headers: {
          Authorization: isSuperAdmin
            ? `Bearer ${traccarBearerToken}`
            : `Bearer ${traccarUserToken}`,
        },
      }
    );

    let responses = {
      userId: userId,
      traccar: traccarResponse.data,
    };

    const insertID = await createCommand(responses);

    responses.id = insertID;

    res.status(200).json({
      status: true,
      message: "Command Created successfully.",
      data: responses,
    });
  } catch (error) {
    console.error(
      "Error in creating Command:",
      error.response ? error.response.data : error.message
    );

    let errorMessage = "Internal Server Error";
    if (error.response?.data?.includes("Duplicate entry ")) {
      errorMessage = `${req.body.email} already exist`;
    } else if (error.response && error.response.status === 401) {
      errorMessage = "Unauthorized";
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.get("/new-commands", async (req, res) => {
  try {
    const commands = await getAllCommands();

    if (!commands) {
      res.status(404).json({
        status: false,
        error: "Commands not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: commands,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-commands/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getCommandById(id);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Command not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-commands/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const result = await getCommandsByUserId(userId);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Commands not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/new-commands/:id", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...Fields } = req.body;
    const id = req.params.id;
    const result = await getCommandById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Command not found",
      });
    }

    const requestData = {
      id: result.traccarId,
      ...Fields,
    };

    const response = await axios.put(
      `http://${process.env.TraccarPort}/api/commands/${requestData.id}`,
      requestData,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    const update = await updateCommandById(id, response.data);

    return res.status(200).json({
      status: true,
      message: "Command Update successfully.",
      data: update,
    });
  } catch (error) {
    console.error("Error in updating command:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.delete("/new-commands/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getCommandById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Command not found",
      });
    }

    const response = await axios.delete(
      `http://${process.env.TraccarPort}/api/commands/${result.traccarId}`,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    if (response.status === 204) {
      await softDeleteCommandById(id);
    }

    return res.status(200).json({
      status: true,
      message: "Command Deleted successfully.",
    });
  } catch (error) {
    console.error("Error in updating command:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

// ______________________________________________________________________________________________________________________

router.post("/new-maintenances", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...restFields } = req.body;

    if (
      userId === undefined ||
      userId === null ||
      isNaN(userId) ||
      typeof userId !== "number"
    ) {
      return res.status(400).json({
        status: false,
        error: "User ID is required.",
      });
    }

    if (isSuperAdmin === undefined) {
      return res.status(400).json({
        status: false,
        error: "isSuperAdmin field is required and must be a boolean.",
      });
    }

    if (!isSuperAdmin && !traccarUserToken) {
      return res.status(400).json({
        status: false,
        error: "Nextop Token is required.",
      });
    }
    const { name, type, start, period, attributes } = restFields;
    const traccarResponse = await axios.post(
      `http://${process.env.TraccarPort}/api/maintenance`,
      {
        name,
        type,
        start,
        period,
        attributes,
      },
      {
        headers: {
          Authorization: isSuperAdmin
            ? `Bearer ${traccarBearerToken}`
            : `Bearer ${traccarUserToken}`,
        },
      }
    );

    let responses = {
      userId: userId,
      traccar: traccarResponse.data,
    };

    const insertID = await createMaintenance(responses);

    responses.id = insertID;

    res.status(200).json({
      status: true,
      message: "Maintenance Created successfully.",
      data: responses,
    });
  } catch (error) {
    console.error(
      "Error in creating Maintenance:",
      error.response ? error.response.data : error.message
    );

    let errorMessage = "Internal Server Error";
    if (error.response?.data?.includes("Duplicate entry ")) {
      errorMessage = `${req.body.email} already exist`;
    } else if (error.response && error.response.status === 401) {
      errorMessage = "Unauthorized";
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.get("/new-maintenances", async (req, res) => {
  try {
    const maintenances = await getAllMaintenances();

    if (!maintenances) {
      res.status(404).json({
        status: false,
        error: "Maintenances not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: maintenances,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-maintenances/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getMaintenanceById(id);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Maintenance not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-maintenances/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const result = await getMaintenancesByUserId(userId);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Maintenances not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/new-maintenances/:id", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...Fields } = req.body;
    const id = req.params.id;
    const result = await getMaintenanceById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Maintenance not found",
      });
    }
    const { name, type, start, period, attributes } = Fields;
    const requestData = {
      id: result.traccarId,
      name,
      type,
      start,
      period,
      attributes,
    };

    const response = await axios.put(
      `http://${process.env.TraccarPort}/api/maintenance/${requestData.id}`,
      requestData,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    const update = await updateMaintenanceById(id, response.data);

    return res.status(200).json({
      status: true,
      message: "Maintenance Update successfully.",
      data: update,
    });
  } catch (error) {
    console.error("Error in updating maintenance:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.delete("/new-maintenances/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getMaintenanceById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Maintenance not found",
      });
    }

    const response = await axios.delete(
      `http://${process.env.TraccarPort}/api/maintenance/${result.traccarId}`,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    if (response.status === 204) {
      await softDeleteMaintenanceById(id);
    }

    return res.status(200).json({
      status: true,
      message: "Maintenance Deleted successfully.",
    });
  } catch (error) {
    console.error("Error in updating maintenance:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

// ______________________________________________________________________________________________________________________

router.post("/new-drivers", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...restFields } = req.body;

    if (
      userId === undefined ||
      userId === null ||
      isNaN(userId) ||
      typeof userId !== "number"
    ) {
      return res.status(400).json({
        status: false,
        error: "User ID is required.",
      });
    }

    if (isSuperAdmin === undefined) {
      return res.status(400).json({
        status: false,
        error: "isSuperAdmin field is required and must be a boolean.",
      });
    }

    if (!isSuperAdmin && !traccarUserToken) {
      return res.status(400).json({
        status: false,
        error: "Nextop Token is required.",
      });
    }
    const { name, uniqueId, attributes } = restFields;
    const traccarResponse = await axios.post(
      `http://${process.env.TraccarPort}/api/drivers`,
      {
        name,
        uniqueId,
        attributes,
      },
      {
        headers: {
          Authorization: isSuperAdmin
            ? `Bearer ${traccarBearerToken}`
            : `Bearer ${traccarUserToken}`,
        },
      }
    );

    let responses = {
      userId: userId,
      traccar: traccarResponse.data,
    };

    const insertID = await createDriver(responses);

    responses.id = insertID;

    res.status(200).json({
      status: true,
      message: "Driver Created successfully.",
      data: responses,
    });
  } catch (error) {
    console.error(
      "Error in creating Driver:",
      error.response ? error.response.data : error.message
    );

    let errorMessage = "Internal Server Error";
    if (error.response?.data?.includes("Duplicate entry ")) {
      errorMessage = `${req.body.email} already exist`;
    } else if (error.response && error.response.status === 401) {
      errorMessage = "Unauthorized";
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.get("/new-drivers", async (req, res) => {
  try {
    const drivers = await getAllDrivers();

    if (!drivers) {
      res.status(404).json({
        status: false,
        error: "Drivers not found",
      });
      return;
    }

    const flattenedDrivers = drivers.map(driver => {
      if (driver.attributes) {
        return { ...driver, ...driver.attributes };
      }
      return driver;
    });

    res.status(200).json({
      status: true,
      data: flattenedDrivers,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-drivers/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getDriverById(id);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Driver not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-drivers/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const result = await getDriversByUserId(userId);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Drivers not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/new-drivers/:id", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...Fields } = req.body;
    const id = req.params.id;
    const result = await getDriverById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Driver not found",
      });
    }

    const { name, uniqueId, attributes } = Fields;
    const requestData = {
      id: result.traccarId,
      name,
      uniqueId,
      attributes,
    };

    const response = await axios.put(
      `http://${process.env.TraccarPort}/api/drivers/${requestData.id}`,
      requestData,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    const update = await updateDriverById(id, response.data);

    return res.status(200).json({
      status: true,
      message: "Driver Update successfully.",
      data: update,
    });
  } catch (error) {
    console.error("Error in updating driver:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.delete("/new-drivers/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getDriverById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Driver not found",
      });
    }

    const response = await axios.delete(
      `http://${process.env.TraccarPort}/api/drivers/${result.traccarId}`,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    if (response.status === 204) {
      await softDeleteDriverById(id);
    }

    return res.status(200).json({
      status: true,
      message: "Driver Deleted successfully.",
    });
  } catch (error) {
    console.error("Error in updating driver:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

// ______________________________________________________________________________________________________________________

router.post("/new-notifications", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...restFields } = req.body;

    if (
      userId === undefined ||
      userId === null ||
      isNaN(userId) ||
      typeof userId !== "number"
    ) {
      return res.status(400).json({
        status: false,
        error: "User ID is required.",
      });
    }

    if (isSuperAdmin === undefined) {
      return res.status(400).json({
        status: false,
        error: "isSuperAdmin field is required and must be a boolean.",
      });
    }

    if (!isSuperAdmin && !traccarUserToken) {
      return res.status(400).json({
        status: false,
        error: "Nextop Token is required.",
      });
    }
    const { type, always, calendarId, commandId, attributes, notificators } =
      restFields;
    const traccarResponse = await axios.post(
      `http://${process.env.TraccarPort}/api/notifications`,
      {
        type,
        always,
        calendarId,
        commandId,
        attributes,
        notificators,
      },
      {
        headers: {
          Authorization: isSuperAdmin
            ? `Bearer ${traccarBearerToken}`
            : `Bearer ${traccarUserToken}`,
        },
      }
    );

    let responses = {
      userId: userId,
      traccar: traccarResponse.data,
    };

    const insertID = await createNotifications(responses);

    responses.id = insertID;

    res.status(200).json({
      status: true,
      message: "Notification Created successfully.",
      data: responses,
    });
  } catch (error) {
    console.error(
      "Error in creating Notification:",
      error.response ? error.response.data : error.message
    );

    let errorMessage = "Internal Server Error";
    if (error.response?.data?.includes("Duplicate entry ")) {
      errorMessage = `${req.body.email} already exist`;
    } else if (error.response && error.response.status === 401) {
      errorMessage = "Unauthorized";
    }

    res.status(error.response ? error.response.status : 500).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.get("/new-notifications", async (req, res) => {
  try {
    const notifications = await getAllNotifications();

    if (!notifications) {
      res.status(404).json({
        status: false,
        error: "Notifications not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: notifications,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-notifications/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getNotificationById(id);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Notification not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-notifications/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const result = await getNotificationsByUserId(userId);

    if (!result) {
      res.status(404).json({
        status: false,
        error: "Notifications not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/new-notifications/:id", async (req, res) => {
  try {
    const { userId, isSuperAdmin, traccarUserToken, ...Fields } = req.body;
    const id = req.params.id;
    const result = await getNotificationById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Notification not found",
      });
    }
    const { type, always, calendarId, commandId, attributes, notificators } =
      Fields;
    const requestData = {
      id: result.traccarId,
      type,
      always,
      calendarId,
      commandId,
      attributes,
      notificators,
    };

    const response = await axios.put(
      `http://${process.env.TraccarPort}/api/notifications/${requestData.id}`,
      requestData,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    const update = await updateNotificationById(id, response.data);

    return res.status(200).json({
      status: true,
      message: "Notification Update successfully.",
      data: update,
    });
  } catch (error) {
    console.error("Error in updating notification:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.delete("/new-notifications/:id", async (req, res) => {
  try {
    const id = req.params.id;
    const result = await getNotificationById(id);

    if (!result) {
      return res.status(404).json({
        status: false,
        error: "Notification not found",
      });
    }

    const response = await axios.delete(
      `http://${process.env.TraccarPort}/api/notification/${result.traccarId}`,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    if (response.status === 204) {
      await softDeleteNotificationById(id);
    }

    return res.status(200).json({
      status: true,
      message: "Notification Deleted successfully.",
    });
  } catch (error) {
    console.error("Error in updating notification:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.get("/notifications/types", async (req, res) => {
  try {
    const response = await axios.get(
      `http://${process.env.TraccarPort}/api/notifications/types`,
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );

    if (response.data) {
      return res.status(200).json({
        status: true,
        data: response.data,
      });
    }
  } catch (error) {
    console.error("Error in updating notification:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.post("/notifications/test", async (req, res) => {
  try {
    const response = await axios.post(
      `http://${process.env.TraccarPort}/api/notifications/test`,
      {},
      {
        headers: {
          Authorization: `Bearer ${traccarBearerToken}`,
        },
      }
    );
    console.log(response);
    // if(response.data){
    return res.status(200).json({
      status: true,
      // data: response.data,
    });
    // }
  } catch (error) {
    console.error("Error in updating notification:", error);

    let errorMessage = "Internal Server Error";
    let statusCode = 500;

    if (error.response.data) {
      statusCode = error.response.status;
      errorMessage = error.response.data;
    }
    if (error.response) {
      if (
        error.response.data &&
        error.response.data.includes("Duplicate entry ")
      ) {
        errorMessage = `${req.body.email} already exists`;
        statusCode = error.response.status;
      } else if (error.response.status === 401) {
        errorMessage = "Unauthorized";
        statusCode = 401;
      }
    }

    return res.status(statusCode).json({
      status: false,
      error: errorMessage,
    });
  }
});

// ______________________________________________________________________________________________________________________

router.post("/new-calculators", async (req, res) => {
  try {
    const { userId, isSuperAdmin,flespiUserToken,traccarUserToken, calc_id, ...restFields } = req.body;

    const flespiResponse = await axios.post(
      `${flespiApiUrl}/calcs?fields=id%2Cname%2Ccid%2Cupdate_delay%2Cmessages_source%2Cversion%2Cintervals_ttl%2Cintervals_rotate%2Cselectors%2Ccounters%2Cvalidate_interval%2Cvalidate_message%2Ctimezone%2Cblocked%2Cdevices_count%2Cintervals_size%2Cupdate_period%2Cupdate_onchange%2Cmetadata`,
      [{ ...restFields }],
      {
        headers: {
          "Content-Type": "application/json",
          Authorization: flespiToken,
        },
      }
    );

    let responses = {
      userId: userId,
      flespi: {
        ...flespiResponse.data.result[0],
      },
    };

    const insertID = await createCalculator(responses);

    responses.id = insertID;
    res.status(200).json({
      status: true,
      message: "Calculator Created successfully.",
      data: responses,
    });
  } catch (error) {
    let errorCode;
    let errorMessage;

    if (error.response) {
      errorCode = error.response.status;
      if (error.response.data.errors) {
        errorMessage = error.response.data.errors;
      } else {
        errorMessage = "Internal server error";
      }
      console.log("Error code:", errorCode);
      console.log("Errors:", errorMessage);
    } else {
      console.log("API Error Message:", error);
      errorCode = 500;
      errorMessage = "Internal server error";
    }

    res.status(errorCode).json({
      status: false,
      errors: errorMessage,
    });
  }
});

router.get("/new-calculators", async (req, res) => {
  try {
    const calculators = await getAllCalculators();

    if (!calculators) {
      res.status(404).json({
        status: false,
        error: "Calculators not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: calculators,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-calculators/:id", async (req, res) => {
  try {
    const calcId = req.params.id;
    const calculator = await getCalculatorById(calcId);

    if (!calculator) {
      res.status(404).json({
        status: false,
        error: "Calculator not found",
      });
      return;
    }

    res.status(200).json({
      status: true,
      data: calculator,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.get("/new-calculators/user/:userId", async (req, res) => {
  try {
    const userId = req.params.userId;
    const calculators = await getCalculatorsByUserId(userId);

    res.status(200).json({
      status: true,
      data: calculators,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

router.put("/new-calculators/:id", async (req, res) => {
  try {
    const calcId = req.params.id;
    const calculator = await getCalculatorById(calcId);

    if (!calculator) {
      res.status(404).json({
        status: false,
        error: "Calculator not found",
      });
      return;
    }

    const { id, flespiId, userId, isSuperAdmin, calc_id, cid, created_at,flespiUserToken,traccarUserToken, version,  ...restFields } = req.body;

    const flespiResponse = await axios.put(
      `${flespiApiUrl}/calcs/${calcId}?fields=id%2Cname%2Ccid%2Cupdate_delay%2Cmessages_source%2Cversion%2Cintervals_ttl%2Cintervals_rotate%2Cselectors%2Ccounters%2Cvalidate_interval%2Cvalidate_message%2Ctimezone%2Cblocked%2Cdevices_count%2Cintervals_size%2Cupdate_period%2Cupdate_onchange%2Cmetadata`,
      { ...restFields },
      {
        headers: {
          "Content-Type": "application/json",
          Authorization: flespiToken,
        },
      }
    );

    // console.log(flespiResponse.data.result[0])

    const update = await updateCalculatorById(calculator.id, flespiResponse.data.result[0]);

    // update.attributes = JSON.parse(update.attributes);
    res.status(200).json({
      status: true,
      message: "Calculator Updated successfully.",
      // data: update,
    });
  } catch (error) {
    let errorCode;
    let errorMessage;

    if (error.response) {
      errorCode = error.response.status;
      if (error.response.data.errors) {
        errorMessage = error.response.data.errors;
      } else {
        errorMessage = "Internal server error";
      }
      console.log("Error code:", errorCode);
      console.log("Errors:", errorMessage);
    } else {
      console.log("API Error Message:", error);
      errorCode = 500;
      errorMessage = "Internal server error";
    }

    res.status(errorCode).json({
      status: false,
      errors: errorMessage,
    });
  }
});

router.delete("/new-calculators/:id", async (req, res) => {
  try {
    const calcId = req.params.id;
    const calculator = await getCalculatorById(calcId);
    if (!calculator) {
      return res.status(404).json({
        status: false,
        error: "Calculator not found",
      });
    }
    const flespiResponse = await axios.delete(`${flespiApiUrl}/calcs/${calcId}`, {
      headers: {
        Authorization: flespiToken,
      },
    });
    if (flespiResponse.status === 200) {
      await softDeleteCalculatorById(calcId);
      return res.status(200).json({
        status: true,
        message: "Calculator deleted successfully.",
      });
    }
  } catch (error) {
    let errorCode = 500;
    let errorMessage = "Internal server error";
    if (error.response) {
      errorCode = error.response.status;
      if (error.response.data.errors) {
        errorMessage = error.response.data.errors;
      } else if (error.response.data.errorMessage) {
        errorMessage = error.response.data.errorMessage;
      }
    } else {
      console.log("API Error Message:", error);
    }
    return res.status(errorCode).json({
      status: false,
      errors: errorMessage,
    });
  }
});


// ______________________________________________________________________________________________________________________


router.get("/calc/1694497/devices/:devices", async (req, res) => {
  const { devices } = req.params;
  const apiUrl = `https://flespi.io/gw/calcs/1694497/devices/${devices}/intervals/all?data=%7B%22reverse%22%3Atrue%2C%22count%22%3A1%7D`;
  const headers = {
    Authorization: `FlespiToken ${flespiToken}`,
  };

  try {
    // console.log(apiUrl);
    const response = await axios.get(apiUrl, { headers });

    res.status(200).json({
      status: true,
      data: response.data.result,
    });
  } catch (error) {
    // console.log(error);
    if (error.response?.data?.errors && Array.isArray(error.response?.data?.errors)) {
      res.status(error.response.status).json({
        status: false,
        errors: error.response?.data?.errors,
      });
    } else {
      res.status(500).json({
        status: false,
        message: `Error: ${error.message}`,
      });
    }
  }
});




// ______________________________________________________________________________________________________________________

router.get("/dispatch", async (req, res) => {
  const apiUrl = `https://flespi.io/gw/devices/all/messages?data=%7B%22reverse%22%3Atrue%2C%22count%22%3A1%2C%22fields%22%3A%22timestamp%2Cdevice.id%2Cdevice.name%2Cident%2Cposition.latitude%2Cposition.longitude%2Crfid.code%2Cmovement.status%22%7D`;
  const headers = {
    Authorization: `FlespiToken ${flespiToken}`,
  };

  try {
    const response = await axios.get(apiUrl, { headers });

    res.status(200).json({
      status: true,
      data: response.data.result,
    });
  } catch (error) {
    if (error.response?.data?.errors && Array.isArray(error.response?.data?.errors)) {
      res.status(error.response.status).json({
        status: false,
        errors: error.response?.data?.errors,
      });
    } else {
      res.status(500).json({
        status: false,
        message: `Error: ${error.message}`,
      });
    }
  }
});

// ______________________________________________________________________________________________________________________


const upload = multer({
  storage: multer.memoryStorage(),
});


function addDeviceAndGroupInfoToData(data, devices, groups) {
  // Create lookup objects for quick access
  const deviceLookup = devices.reduce((lookup, device) => {
    lookup[device.flespiId] = device;
    return lookup;
  }, {});

  const groupLookup = groups.reduce((lookup, group) => {
    lookup[group.traccarId] = group;
    return lookup;
  }, {});

  return data.map(item => {
    const device = deviceLookup[item.device_id];
    const group = device ? groupLookup[device.groupId] : null;

    return {
      ...item,
      device_name: device ? device.name : null,
      group_id: group ? group.id : null,
      group_name: group ? group.name : null,
      created_at: item.created_at ? formatTimeISO(item.created_at) : null,
      updated_at: item.updated_at ? formatTimeISO(item.updated_at) : null,
      accepted_at: item.accepted_at ? formatTimeISO(item.accepted_at) : null,
      rejected_at: item.rejected_at ? formatTimeISO(item.rejected_at) : null,
      completed_at: item.completed_at ? formatTimeISO(item.completed_at) : null,
    };
  });
}

// Create task
router.post("/dispatch/tasks", upload.array("pictures"), async (req, res) => {
  const { user_id, device_id, address, place_id, message, lat, lng } = req.body;

  const missingFields = [];
  if (!user_id) missingFields.push('user_id');
  if (!device_id) missingFields.push('device_id');
  if (!address) missingFields.push('address');
  if (!place_id) missingFields.push('place_id');
  if (!lat) missingFields.push('lat');
  if (!lng) missingFields.push('lng');
  if (!message) missingFields.push('message');

  if (missingFields.length > 0) {
    return res.status(400).json({ status: false, error: `Missing fields: ${missingFields.join(', ')}` });
  }

  // const user = await getUserById(user_id);
  // const device = await getDeviceById(device_id);

  // if (!user) {
  //   return res.status(404).json({ status: false, error: 'User not found' });
  // }

  // if (!device) {
  //   return res.status(404).json({ status: false, error: 'Device not found' });
  // }

  const s3Client = new S3({
    region: "US-central",
    credentials: {
      accessKeyId: process.env.CONTABO_ACCESS_KEY_ID,
      secretAccessKey: process.env.CONTABO_SECRET_ACCESS_KEY,
    },
    endpoint: process.env.CONTABO_ENDPOINT,
    forcePathStyle: true,
  });

  try {

    const uploadPromises = req.files.map(async (file) => {
      const timestamp = Date.now();
      const uploadResult = await s3Client.putObject({
        Bucket: 'task-images',
        Key: `${user_id}-${device_id}-${timestamp}-${file.originalname}`,
        ACL: "public-read",
        Body: file.buffer,
      }); 

      console.log("tasks image uploadResult", uploadResult);

      return `${process.env.CONTABO_ENDPOINT}/${process.env.CONTABO_BUCKET_ADDRESS}:task-images/${user_id}-${device_id}-${timestamp}-${file.originalname}`;
    });

    const uploadedPictures = await Promise.all(uploadPromises);

    const taskData = {
      user_id,
      device_id,
      address,
      place_id,
      lat,
      lng,
      message,
      pictures: uploadedPictures.join(','),
      status: 'pending'
    };

    const taskId = await createTask(taskData);

    const response = {
      id: taskId,
      user_id,
      device_id,
      address,
      place_id,
      lat,
      lng,
      message,
      pictures: uploadedPictures, 
    };

    res.status(201).json({ status: true, data: response });
  } catch (error) {
    console.error("Error creating task:", error);
    res.status(500).json({ status: false, error: 'An error occurred while creating the task' });
  }
});

// Create task search history
router.post("/dispatch/tasks-sh", async (req, res) => {
  const { user_id, address, place_id, lat, lng, radius } = req.body;

  const missingFields = [];
  if (!user_id) missingFields.push('user_id');
  if (!address) missingFields.push('address');
  if (!place_id) missingFields.push('place_id');
  if (!lat) missingFields.push('lat');
  if (!lng) missingFields.push('lng');
  if (!radius) missingFields.push('radius');

  if (missingFields.length > 0) {
    return res.status(400).json({ status: false, error: `Missing fields: ${missingFields.join(', ')}` });
  }
  try {

    const sh = {
      user_id,
      address,
      place_id,
      lat,
      lng,
      radius,
    };

    const taskSHId = await createTaskSH(sh);

    const response = {
      id: taskSHId,
      user_id,
      address,
      place_id,
      lat,
      lng,
      radius,
    };

    res.status(201).json({ status: true, data: response });
  } catch (error) {
    console.error("Error creating task:", error);
    res.status(500).json({ status: false, error: 'An error occurred while creating the task history' });
  }
});

// Get all tasks
router.get("/dispatch/tasks", async (req, res) => {
  try {
    const tasks = await getAllTasks();
    const devices = await getAllDevices();
    const groups = await getAllGroups();    

    // Add device and group info to tasks
    const enhancedTasks = addDeviceAndGroupInfoToData(tasks, devices, groups);

    res.status(200).json({ status: true, data: enhancedTasks });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: false, error: 'An error occurred while fetching tasks' });
  }
});

// Get tasks by user ID
router.get("/dispatch/tasks/user/:userId", async (req, res) => {
  const { userId } = req.params;
  
  try {
    const tasks = await getTasksByUserId(userId);
    const devices = await getAllDevices();
    const groups = await getAllGroups();    

    // Add device and group info to tasks
    const enhancedTasks = addDeviceAndGroupInfoToData(tasks, devices, groups);

    res.status(200).json({ status: true, data: enhancedTasks });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: false, error: 'An error occurred while fetching tasks' });
  }
});

// Get tasks search history by user ID
router.get("/dispatch/tasks-sh/:userId", async (req, res) => {
  const { userId } = req.params;
  
  try {
    const tasks = await getTasksSHByUserId(userId); 

    const transformedTasks = tasks.map(task => {
      return {
        ...task,
        time: formatTimeISOGMTMINUS5(task.created_at),
      };
    });

    res.status(200).json({ status: true, data: transformedTasks });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: false, error: 'An error occurred while fetching tasks' });
  }
});

// Get task by ID
router.get("/dispatch/tasks/:id", async (req, res) => {
  const { id } = req.params;
  
  try {
    const task = await getTaskById(id);
    if (!task) {
      return res.status(404).json({ status: false, error: 'Task not found' });
    }
    const devices = await getAllDevices();
    const groups = await getAllGroups();    

    // Add device and group info to the task
    const enhancedTask = addDeviceAndGroupInfoToData([task], devices, groups)[0];

    res.status(200).json({ status: true, data: enhancedTask });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: false, error: 'An error occurred while fetching the task' });
  }
});

// Get tasks by device ID
router.get("/dispatch/tasks/device/:deviceId", async (req, res) => {
  const { deviceId } = req.params;
  
  try {
    const tasks = await getTasksByDeviceId(deviceId);
    const devices = await getAllDevices();
    const groups = await getAllGroups();    

    // Add device and group info to tasks
    const enhancedTasks = addDeviceAndGroupInfoToData(tasks, devices, groups);

    res.status(200).json({ status: true, data: enhancedTasks });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: false, error: 'An error occurred while fetching tasks' });
  }
});



function verifyToken(req, res, next) {
  const token = req.headers['authorization'];
  if (!token) {
    return res.status(403).json({ status: false, error: 'No token provided' });
  }

  jwt.verify(token, secretKey, (err, decoded) => {
    if (err) {
      return res.status(500).json({ status: false, error: 'Unauthorized' });
    }
    req.deviceId = decoded.id;
    next();
  });
}

async function getDevice(req, res, next) {
  try {
    const device = await getDeviceById(req.deviceId);
    if (!device) {
      return res.status(404).json({ status: false, error: 'Device not found' });
    }
    // console.log(device);
    req.device = device;
    next();
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: false, error: 'Internal Server Error' });
  }
}

router.get("/new-device", verifyToken, getDevice, async (req, res) => {
  try {
    const device = req.device;
    res.status(200).json({ status: true, data: device });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: false, error: 'Internal Server Error' });
  }
});


const validStatuses = ['accepted', 'rejected', 'completed'];

router.put("/dispatch/tasks", verifyToken, getDevice, async (req, res) => {
  const device = req.device;

  const { id, status, description } = req.body;

  if (!id || !status) {
    return res.status(400).json({ status: false, error: 'id and status are required' });
  }
  
  if (!validStatuses.includes(status)) {
    return res.status(400).json({ status: false, error: `Invalid status. Valid statuses are: ${validStatuses.join(', ')}` });
  }

  try {
    const success = await updateTaskStatusByIdAndDeviceId(id, device.flespiId, status, description);

    if (success) {
      res.status(200).json({ status: true, message: 'Task status updated successfully' });
    } else {
      res.status(404).json({ status: false, error: 'No tasks found with the provided id and deviceId' });
    }
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: false, error: 'Internal Server Error' });
  }
});

// // API endpoint to get place details
// router.get('/place-details', async (req, res) => {
//   const { placeId } = req.query;

//   if (!placeId) {
//     return res.status(400).json({ error: 'placeId is required' });
//   }

//   try {
//     const response = await axios.get(`https://maps.googleapis.com/maps/api/place/details/json`, {
//       params: {
//         placeid: placeId,
//         key: process.env.GOOGLE_MAP_API
//       }
//     });

//     res.json(response.data);
//   } catch (error) {
//     console.error('Error fetching place details:', error);
//     res.status(500).json({ error: 'Error fetching place details' });
//   }
// });




// ______________________________________________________________________________________________________________________



// router.get("/dispatch", async (req, res) => {
//   const apiUrl = `https://flespi.io/gw/devices/all/telemetry/position.latitude%2Cposition.longitude%2Cdevice.name%2Cdevice.time`;
//   const apiUrl2 = `https://flespi.io/gw/devices/all/messages?data=%7B%22reverse%22%3Atrue%2C%22count%22%3A1%2C%22fields%22%3A%22timestamp%2Cdevice.id%2Cdevice.name%2Cident%2Cposition.latitude%2Cposition.longitude%2Crfid.code%2Cmovement.status%22%7D`;
//   const headers = {
//     Authorization: `FlespiToken ${flespiToken}`,
//   };

//   try {
//     const response = await axios.get(apiUrl, { headers });
//     const response2 = await axios.get(apiUrl2, { headers });

    
// function matchData(data, data2) {
//   const matchedData = [];

//   data.forEach(item1 => {
//       const matchedItem = data2.find(item2 => item1.id === item2["device.id"]);
//       if (matchedItem) {
//           const { "ident": ident, "rfid.code": rfidCode, "movement.status": movementStatus } = matchedItem;
//           matchedData.push({ ...item1, ident, rfidCode, "movement.status": movementStatus });
//       }
//   });

//   return matchedData;
// }

// const matchedData = matchData(response.data.result, response2.data.result);

//     res.status(200).json({
//       status: true,
//       telemetry: response.data.result,
//       message: response2.data.result,
//       data: matchedData,
//     });
//   } catch (error) {
//     if (error.response?.data?.errors && Array.isArray(error.response?.data?.errors)) {
//       res.status(error.response.status).json({
//         status: false,
//         errors: error.response?.data?.errors,
//       });
//     } else {
//       res.status(500).json({
//         status: false,
//         message: `Error: ${error.message}`,
//       });
//     }
//   }
// });

// ______________________________________________________________________________________________________________________


router.get("/devices/:device/messages/:from/:to", async (req, res) => {
  const { device, from, to } = req.params;
  // const apiUrl = `https://flespi.io/gw/devices/${device}/messages?data=%7B%22from%22%3A${from}%2C%22to%22%3A${to}%2C%22fields%22%3A%22position.latitude%2Cposition.longitude%2Cposition.direction%22%7D`;
  const apiUrl = `https://flespi.io/gw/devices/${device}/messages?data=%7B%22from%22%3A${from}%2C%22to%22%3A${to}%7D`;
  const headers = {
    Authorization: `FlespiToken ${flespiToken}`,
  };

  try {
    const response = await axios.get(apiUrl, { headers });

    res.status(200).json({
      status: true,
      data: response.data.result,
    });
  } catch (error) {
    // console.log(error);
    if (error.response?.data?.errors && Array.isArray(error.response?.data?.errors)) {
      res.status(error.response.status).json({
        status: false,
        errors: error.response?.data?.errors,
      });
    } else {
      res.status(500).json({
        status: false,
        message: `Error: ${error.message}`,
      });
    }
  }
});

router.get("/devices/:device/messages/:from/:to/CRT", async (req, res) => {
  const { device, from, to } = req.params;
  const apiUrl = `https://flespi.io/gw/devices/${device}/messages?data=%7B%22from%22%3A${from}%2C%22to%22%3A${to}%2C%22reverse%22%3Atrue%2C%22count%22%3A1%2C%22fields%22%3A%22device.name%2Cdevice.id%2Cident%2Cposition.latitude%2Cposition.longitude%2Crfid.code%2Ctimestamp%2C%22%7D`;
  const headers = {
    Authorization: `FlespiToken ${flespiToken}`,
  };

  try {
    const response = await axios.get(apiUrl, { headers });

    res.status(200).json({
      status: true,
      data: response.data.result,
    });
  } catch (error) {
    // console.log(error);
    if (error.response?.data?.errors && Array.isArray(error.response?.data?.errors)) {
      res.status(error.response.status).json({
        status: false,
        errors: error.response?.data?.errors,
      });
    } else {
      res.status(500).json({
        status: false,
        message: `Error: ${error.message}`,
      });
    }
  }
});


// Create Point search history
router.post("/point/sh", async (req, res) => {
  const { user_id, search_time, address, place_id, lat, lng, radius } = req.body;

  const missingFields = [];
  if (!user_id) missingFields.push('user_id');
  if (!search_time) missingFields.push('search_time');
  if (!address) missingFields.push('address');
  if (!place_id) missingFields.push('place_id');
  if (!lat) missingFields.push('lat');
  if (!lng) missingFields.push('lng');
  if (!radius) missingFields.push('radius');

  if (missingFields.length > 0) {
    return res.status(400).json({ status: false, error: `Missing fields: ${missingFields.join(', ')}` });
  }
  try {

    const sh = {
      user_id,
      search_time,
      address,
      place_id,
      lat,
      lng,
      radius,
    };

    const pointSHId = await createPointSH(sh);

    const response = {
      id: pointSHId,
      user_id,
      search_time,
      address,
      place_id,
      lat,
      lng,
      radius,
    };

    res.status(201).json({ status: true, data: response });
  } catch (error) {
    console.error("Error creating point:", error);
    res.status(500).json({ status: false, error: 'An error occurred while creating the point history' });
  }
});

// Get points search history by user ID
router.get("/point/sh/:userId", async (req, res) => {
  const { userId } = req.params;
  
  try {
    const points = await getPointsSHByUserId(userId); 

    const transformedPoints = points.map(task => {
      return {
        ...task,
        search_time_formated: formatTimeISO(task.search_time),
        time: formatTimeISOGMTMINUS5(task.created_at),
      };
    });

    res.status(200).json({ status: true, data: transformedPoints });
  } catch (error) {
    console.error(error);
    res.status(500).json({ status: false, error: 'An error occurred while fetching points' });
  }
});



// ______________________________________________________________________________________________________________________

const fetchSnapshots = async () => {
  try {
    const snapshotsApi = await axios.post(`${baseURL}/new-snapshots`, null, {
      timeout: 15000,
    });

    console.log(snapshotsApi.data);
  } catch (error) {
    console.error("Error snapshots:", error.message);
  }
};

// fetchSnapshots();

const job2 = schedule.scheduleJob("1 1 * * *", async () => {
  fetchSnapshots();
});

module.exports = router;
