const express = require("express");
const router = express.Router();
const axios = require("axios");
const { Device, getAllDevices, getDevicesByIMEI } = require("../models/device");
const {
  // getCalcDailyReports,
  getAllDevicesCalcs,
  insertOrUpdateCalcDailyReports,
  getCalcDailyReportsByDeviceId,
  getCalcDailyReports,
} = require("../models/calc_daily_reports");
const {
  insertOrUpdateCalcStopDectector,
  // getCalcStopDectector,
  getCalcStopDectectorByDeviceId,
} = require("../models/calc_stop_detector");
const {
  insertOrUpdateCalcRefuel,
  getCalcRefuel,
  getCalcRefuelByDeviceId,
} = require("../models/calc_refuel");
const {
  insertOrUpdateCalcFuelConsumed,
  getCalcFuelConsumed,
  getCalcFuelConsumedByDeviceId,
} = require("../models/calc_fuel_consumed");
const {
  insertOrUpdateCalcChecklistOpt,
  getCalcChecklistOpt,
  getCalcChecklistOptByDeviceId,
} = require("../models/calc_checklist_operators");
const {
  insertDrivers,
  getDrivers,
  getAllDriversIds,
  updateDriver,
  deleteDriver,
} = require("../models/drivers");
const {
  extractUniqueDriverIds,
  findAssignedAndNotAssignedDrivers,
  transformCanDataArray,
  matchDataWithDeviceCalcs,
  getAllUniqueIds,
} = require("../helper/common");
const {
  insertOrUpdateCalcParadas,
  getCalcParadas,
  getCalcParadasByDeviceId,
} = require("../models/calc_paradas");
const {
  insertOrUpdateCalcDailyTravelLogs,
  getCalcDailyTravelLogs,
  getCalcDailyTravelLogsByDeviceId,
} = require("../models/calc_daily_travel_logs");
const {
  insertSubAccount,
  getSubAccounts,
  getSubAccountById,
  insertFlespiTracaarAcc,
} = require("../models/sub_account");
const {
  insertSubAccountToken,
  getFlespiTokenById,
} = require("../models/sub_account_token");
const {
  insertTraccarToken,
  getTraccarToken,
  getTraccarTokenById,
} = require("../models/traccar_token");
const {
  insertOrUpdateDeviceData,
  getDeviceData,
} = require("../models/create_devices");
const { getMatchingDevices } = require("../utils/commons");

// _____________________________________________________________________________________________________________________


router.post("/calc-daily-reports", async (req, res) => {
  try {
    const calculators = [{ id: 1675142 }];

    const allProcessedData = [];

    for (const calc of calculators) {
      const apiUrl = `https://flespi.io/gw/calcs/${calc.id}/devices/all/intervals/all?data=%7B%7D`;
      const headers = {
        Authorization: `FlespiToken ${process.env.FlespiToken}`,
      };

      try {
        const response = await axios.get(apiUrl, { headers });

        if (response.status === 200) {
          const dataWithDeviceName = response.data.result.map((dataItem) => {
            dataItem.calc_id = calc.id;
            return dataItem;
          });

          allProcessedData.push(...dataWithDeviceName);
        } else {
          console.log("API response error:", response.status, response.data);
          res.status(500).json({
            status: false,
            error: "Failed to retrieve data from the API",
          });
          return;
        }
      } catch (apiError) {
        console.log("API request error:", apiError);
        res.status(500).json({
          status: false,
          error: "An error occurred while fetching data from the API",
        });
        return;
      }
    }

    await insertOrUpdateCalcDailyReports(allProcessedData);

    res.status(200).json({
      status: true,
      data: allProcessedData,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

router.get("/calc-daily-reports", async (req, res) => {
  try {
    const AllDevicesandCalcs = await new Promise((resolve, reject) => {
      getAllDevicesCalcs((error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const AllCalcReports = await new Promise((resolve, reject) => {
      getCalcDailyReports((error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const dataWithNames = matchDataWithDeviceCalcs(
      AllCalcReports,
      AllDevicesandCalcs
    );

    res.status(200).json({
      status: true,
      data: dataWithNames,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

router.post("/get-calc-daily-reports", async (req, res) => {
  const deviceId = req.body;
  if (!deviceId || !Array.isArray(deviceId)) {
    res.status(400).json({
      status: false,
      error: "Invalid or missing deviceId in the request body",
    });
    return;
  }

  try {
    const AllDevicesandCalcs = await new Promise((resolve, reject) => {
      getAllDevicesCalcs((error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const AllCalcReports = await new Promise((resolve, reject) => {
      getCalcDailyReportsByDeviceId(deviceId, (error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const dataWithNames = matchDataWithDeviceCalcs(
      AllCalcReports,
      AllDevicesandCalcs
    );

    res.status(200).json({
      status: true,
      data: dataWithNames,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

// _____________________________________________________________________________________________________________________

router.post("/calc-stop-detector", async (req, res) => {
  try {
    getAllDevicesCalcs(async (error, results) => {
      if (error) {
        res.status(500).json({
          status: false,
          error: "An error occurred while fetching data",
        });
      } else {
        const data = results;
        function filterDataByCalcId(data, calcIds) {
          return data.filter((entry) => calcIds.includes(entry.calc_id));
        }
        const calcIdsToFilter = [1673604];
        const filteredData = filterDataByCalcId(data, calcIdsToFilter);
        const processedData = [];

        for (const entry of filteredData) {
          try {
            const apiUrl = `https://flespi.io/gw/calcs/${entry.calc_id}/devices/${entry.device_id}%2C/intervals/all?data=%7B%7D`;
            const headers = {
              Authorization: `FlespiToken ${process.env.FlespiToken}`,
            };

            const response = await axios.get(apiUrl, { headers });

            if (response.status === 200) {
              const devicesData = response.data.result;
              for (const deviceEntry of devicesData) {
                processedData.push({
                  calc_name: entry.name,
                  calc_id: entry.calc_id,
                  device_name: entry.device_name,
                  device_id: entry.device_id,
                  data: deviceEntry,
                });
              }
            } else {
              res.status(500).json({
                status: false,
                error: "Failed to retrieve data from the API",
              });
              return;
            }
          } catch (error) {
            // console.error("Error:", error);
            res.status(500).json({
              status: false,
              error: "An error occurred while fetching and storing data",
            });
            return;
          }
        }

        function filterDataWithId(processedData) {
          return processedData.filter((entry) => entry.data && entry.data.id);
        }

        const filteredProdessedData = filterDataWithId(processedData);
        await insertOrUpdateCalcStopDectector(filteredProdessedData);

        res.status(200).json({
          status: true,
          data: filteredProdessedData,
        });
      }
    });
  } catch (error) {
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

router.get("/calc-stop-detector", (req, res) => {
  getCalcStopDectector((error, results) => {
    if (error) {
      res.status(500).json({
        status: false,
        error: "An error occurred while fetching data",
      });
    } else {
      res.status(200).json({
        status: true,
        data: results,
      });
    }
  });
});

// _____________________________________________________________________________________________________________________

router.post("/calc-refuel", async (req, res) => {
  try {
    const calculators = [{ id: 1674670 }];

    const allProcessedData = [];

    for (const calc of calculators) {
      const apiUrl = `https://flespi.io/gw/calcs/${calc.id}/devices/all/intervals/all?data=%7B%7D`;
      const headers = {
        Authorization: `FlespiToken ${process.env.FlespiToken}`,
      };

      try {
        const response = await axios.get(apiUrl, { headers });

        if (response.status === 200) {
          const dataWithDeviceName = response.data.result.map((dataItem) => {
            dataItem.calc_id = calc.id;
            return dataItem;
          });

          allProcessedData.push(...dataWithDeviceName);
        } else {
          console.log("API response error:", response.status, response.data);
          res.status(500).json({
            status: false,
            error: "Failed to retrieve data from the API",
          });
          return;
        }
      } catch (apiError) {
        console.log("API request error:", apiError);
        res.status(500).json({
          status: false,
          error: "An error occurred while fetching data from the API",
        });
        return;
      }
    }

    await insertOrUpdateCalcRefuel(allProcessedData);

    res.status(200).json({
      status: true,
      data: allProcessedData,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

router.post("/get-calc-refuel", async (req, res) => {
  const deviceId = req.body;
  if (!deviceId || !Array.isArray(deviceId)) {
    res.status(400).json({
      status: false,
      error: "Invalid or missing deviceId in the request body",
    });
    return;
  }

  try {
    const AllDevicesandCalcs = await new Promise((resolve, reject) => {
      getAllDevicesCalcs((error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const AllCalcReports = await new Promise((resolve, reject) => {
      getCalcRefuelByDeviceId(deviceId, (error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const dataWithNames = matchDataWithDeviceCalcs(
      AllCalcReports,
      AllDevicesandCalcs
    );

    res.status(200).json({
      status: true,
      data: dataWithNames,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

// _____________________________________________________________________________________________________________________

router.post("/calc-fuel-consumed", async (req, res) => {
  try {
    const calculators = [{ id: 1674833 }];

    const allProcessedData = [];

    for (const calc of calculators) {
      const apiUrl = `https://flespi.io/gw/calcs/${calc.id}/devices/all/intervals/all?data=%7B%7D`;
      const headers = {
        Authorization: `FlespiToken ${process.env.FlespiToken}`,
      };

      try {
        const response = await axios.get(apiUrl, { headers });

        if (response.status === 200) {
          const dataWithDeviceName = response.data.result.map((dataItem) => {
            dataItem.calc_id = calc.id;
            return dataItem;
          });

          allProcessedData.push(...dataWithDeviceName);
        } else {
          console.log("API response error:", response.status, response.data);
          res.status(500).json({
            status: false,
            error: "Failed to retrieve data from the API",
          });
          return;
        }
      } catch (apiError) {
        console.log("API request error:", apiError);
        res.status(500).json({
          status: false,
          error: "An error occurred while fetching data from the API",
        });
        return;
      }
    }

    await insertOrUpdateCalcFuelConsumed(allProcessedData);

    res.status(200).json({
      status: true,
      data: allProcessedData,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

router.post("/get-calc-fuel-consumed", async (req, res) => {
  const deviceId = req.body;
  if (!deviceId || !Array.isArray(deviceId)) {
    res.status(400).json({
      status: false,
      error: "Invalid or missing deviceId in the request body",
    });
    return;
  }

  try {
    const AllDevicesandCalcs = await new Promise((resolve, reject) => {
      getAllDevicesCalcs((error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const AllCalcReports = await new Promise((resolve, reject) => {
      getCalcFuelConsumedByDeviceId(deviceId, (error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const dataWithNames = matchDataWithDeviceCalcs(
      AllCalcReports,
      AllDevicesandCalcs
    );

    res.status(200).json({
      status: true,
      data: dataWithNames,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

// _____________________________________________________________________________________________________________________

router.post("/calc-checklist-operators", async (req, res) => {
  try {
    const calculators = [{ id: 1674836 }];

    const allProcessedData = [];

    for (const calc of calculators) {
      const apiUrl = `https://flespi.io/gw/calcs/${calc.id}/devices/all/intervals/all?data=%7B%7D`;
      const headers = {
        Authorization: `FlespiToken ${process.env.FlespiToken}`,
      };

      try {
        const response = await axios.get(apiUrl, { headers });

        if (response.status === 200) {
          const dataWithDeviceName = response.data.result.map((dataItem) => {
            dataItem.calc_id = calc.id;
            return dataItem;
          });

          allProcessedData.push(...dataWithDeviceName);
        } else {
          console.log("API response error:", response.status, response.data);
          res.status(500).json({
            status: false,
            error: "Failed to retrieve data from the API",
          });
          return;
        }
      } catch (apiError) {
        console.log("API request error:", apiError);
        res.status(500).json({
          status: false,
          error: "An error occurred while fetching data from the API",
        });
        return;
      }
    }

    await insertOrUpdateCalcChecklistOpt(allProcessedData);

    res.status(200).json({
      status: true,
      data: allProcessedData,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});


router.post("/get-calc-checklist-operators", async (req, res) => {
  const deviceId = req.body;
  if (!deviceId || !Array.isArray(deviceId)) {
    res.status(400).json({
      status: false,
      error: "Invalid or missing deviceId in the request body",
    });
    return;
  }

  try {
    const AllDevicesandCalcs = await new Promise((resolve, reject) => {
      getAllDevicesCalcs((error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const AllCalcReports = await new Promise((resolve, reject) => {
      getCalcChecklistOptByDeviceId(deviceId, (error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const dataWithNames = matchDataWithDeviceCalcs(
      AllCalcReports,
      AllDevicesandCalcs
    );

    res.status(200).json({
      status: true,
      data: dataWithNames,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

// ______________________________________________________________________________________________________________________

router.post("/drivers", async (req, res) => {
  try {
    const data = req.body;
    await insertDrivers(data);
    res.status(201).json({
      status: true,
      message: "Driver added successfully",
      data: data,
    });
  } catch (error) {
    console.error(error);
    res.status(500).json({
      status: false,
      error: "Internal Server Error",
      data: null,
    });
  }
});

router.get("/drivers", (req, res) => {
  getDrivers((error, results) => {
    if (error) {
      res.status(500).json({
        status: false,
        error: "An error occurred while fetching data",
      });
    } else {
      res.status(200).json({
        status: true,
        data: results,
      });
    }
  });
});

router.put("/drivers/:id", (req, res) => {
  const driverId = req.params.id;
  const updatedData = req.body;

  updateDriver(driverId, updatedData, (error, result) => {
    if (error) {
      res.status(500).json({
        status: false,
        error: "An error occurred while updating the driver",
      });
    } else {
      res.status(200).json({
        status: true,
        message: "Driver updated successfully",
      });
    }
  });
});

router.delete("/drivers/:id", (req, res) => {
  const driverId = req.params.id;

  deleteDriver(driverId, (error, result) => {
    if (error) {
      res.status(500).json({
        status: false,
        error: "An error occurred while deleting the driver",
      });
    } else {
      res.status(200).json({
        status: true,
        message: "Driver deleted successfully",
      });
    }
  });
});

// ______________________________________________________________________________________________________________________

router.get("/all-drivers", (req, res) => {
  getAllDriversIds((error, results) => {
    if (error) {
      res.status(500).json({
        status: false,
        error: "An error occurred while fetching data",
      });
    } else {
      const uniqueDriverIds = extractUniqueDriverIds(results);
      getDrivers((error, results) => {
        if (error) {
          res.status(500).json({
            status: false,
            error: "An error occurred while fetching data",
          });
        } else {
          const data = findAssignedAndNotAssignedDrivers(
            results,
            uniqueDriverIds
          );
          // const data = {
          //   results,
          //   uniqueDriverIds
          // };
          res.status(200).json({
            status: true,
            data: data,
          });
        }
      });
    }
  });
});

// ______________________________________________________________________________________________________________________

router.post("/calc-paradas", async (req, res) => {
  try {
    const calculators = [{ id: 1674971 }];

    const allProcessedData = [];

    for (const calc of calculators) {
      const apiUrl = `https://flespi.io/gw/calcs/${calc.id}/devices/all/intervals/all?data=%7B%7D`;
      const headers = {
        Authorization: `FlespiToken ${process.env.FlespiToken}`,
      };

      try {
        const response = await axios.get(apiUrl, { headers });

        if (response.status === 200) {
          const dataWithDeviceName = response.data.result.map((dataItem) => {
            dataItem.calc_id = calc.id;
            return dataItem;
          });

          allProcessedData.push(...dataWithDeviceName);
        } else {
          console.log("API response error:", response.status, response.data);
          res.status(500).json({
            status: false,
            error: "Failed to retrieve data from the API",
          });
          return;
        }
      } catch (apiError) {
        console.log("API request error:", apiError);
        res.status(500).json({
          status: false,
          error: "An error occurred while fetching data from the API",
        });
        return;
      }
    }

    await insertOrUpdateCalcParadas(allProcessedData);

    res.status(200).json({
      status: true,
      data: allProcessedData,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});


router.post("/get-calc-paradas", async (req, res) => {
  const deviceId = req.body;
  if (!deviceId || !Array.isArray(deviceId)) {
    res.status(400).json({
      status: false,
      error: "Invalid or missing deviceId in the request body",
    });
    return;
  }

  try {
    const AllDevicesandCalcs = await new Promise((resolve, reject) => {
      getAllDevicesCalcs((error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const AllCalcReports = await new Promise((resolve, reject) => {
      getCalcParadasByDeviceId(deviceId, (error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const dataWithNames = matchDataWithDeviceCalcs(
      AllCalcReports,
      AllDevicesandCalcs
    );

    res.status(200).json({
      status: true,
      data: dataWithNames,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

// ______________________________________________________________________________________________________________________

router.get("/can-real-time/:deviceId", async (req, res) => {
  try {
    const deviceId = req.params.deviceId;

    const apiUrl = `https://flespi.io/gw/devices/${deviceId}?fields=configuration%2Cid%2Cname%2Cdevice_type_id%2Cdevice_type_name%2Cprotocol_id%2Cprotocol_name%2Cmedia_size%2Cmedia_ttl%2Cmedia_blocked%2Cconnected%2Ccid%2Ctelemetry`;
    const headers = {
      Authorization: `FlespiToken ${process.env.FlespiToken}`,
    };

    try {
      const response = await axios.get(apiUrl, { headers });

      if (response.status === 200) {
        const result = response.data.result;
        const transformedCanDataArray = transformCanDataArray(result);
        res.status(200).json({
          status: true,
          data: transformedCanDataArray,
        });
      } else {
        console.log("API response error:", response.status, response.data);
        res.status(500).json({
          status: false,
          error: "Failed to retrieve data from the API",
        });
      }
    } catch (apiError) {
      console.log("API request error:", apiError);
      res.status(500).json({
        status: false,
        error: "An error occurred while fetching data from the API",
      });
    }
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

// ______________________________________________________________________________________________________________________

router.post("/calc-daily-log-travel", async (req, res) => {
  try {
    const calculators = [{ id: 1675136 }];

    const allProcessedData = [];

    for (const calc of calculators) {
      const apiUrl = `https://flespi.io/gw/calcs/${calc.id}/devices/all/intervals/all`;
      const headers = {
        Authorization: `FlespiToken ${process.env.FlespiToken}`,
      };

      try {
        const response = await axios.get(apiUrl, { headers });

        if (response.status === 200) {
          const dataWithDeviceName = response.data.result.map((dataItem) => {
            dataItem.calc_id = calc.id;
            return dataItem;
          });

          allProcessedData.push(...dataWithDeviceName);
        } else {
          console.log("API response error:", response.status, response.data);
          res.status(500).json({
            status: false,
            error: "Failed to retrieve data from the API",
          });
          return;
        }
      } catch (apiError) {
        console.log("API request error:", apiError);
        res.status(500).json({
          status: false,
          error: "An error occurred while fetching data from the API",
        });
        return;
      }
    }

    await insertOrUpdateCalcDailyTravelLogs(allProcessedData);

    res.status(200).json({
      status: true,
      data: allProcessedData,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});


router.post("/get-calc-daily-log-travel", async (req, res) => {
  const deviceId = req.body;
  if (!deviceId || !Array.isArray(deviceId)) {
    res.status(400).json({
      status: false,
      error: "Invalid or missing deviceId in the request body",
    });
    return;
  }

  try {
    const AllDevicesandCalcs = await new Promise((resolve, reject) => {
      getAllDevicesCalcs((error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const AllCalcReports = await new Promise((resolve, reject) => {
      getCalcDailyTravelLogsByDeviceId(deviceId, (error, results) => {
        if (error) {
          reject(error);
        } else {
          resolve(results);
        }
      });
    });

    const dataWithNames = matchDataWithDeviceCalcs(
      AllCalcReports,
      AllDevicesandCalcs
    );

    res.status(200).json({
      status: true,
      data: dataWithNames,
    });
  } catch (error) {
    console.log("Unexpected error:", error);
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

// ______________________________________________________________________________________________________________________


router.post("/subaccount", async (req, res) => {
  try {
    const { created_by, ...restFields } = req.body;

    // Traccar API call
    const traccarResponse = await axios.post(
      `http://${process.env.TraccarPort}/api/users`,
      {
        ...restFields,
      },
      {
        headers: { Authorization: `Bearer ${process.env.TraccarToken}` },
      }
    );

    // Flespi API call
    const flespiResponse = await axios.post(
      "https://flespi.io/platform/subaccounts",
      [{ name: req.body.name }],
      {
        headers: { Authorization: `FlespiToken ${process.env.FlespiToken}` },
      }
    );

    // Data preparation
    const data = {
      created_by,
      flespi: flespiResponse.data.result[0],
      traccar: { ...traccarResponse.data, password: req.body.password },
    };

    // Database insertion
    const insertionStatus = await insertFlespiTracaarAcc(data);

    if (insertionStatus) {
      res.status(200).json({ status: true, data });
    } else {
      res
        .status(500)
        .json({
          status: false,
          error: "Failed to insert data into the database",
        });
    }
  } catch (error) {
    if (error.response && error.response.status === 400) {
      console.log(error);
      res
        .status(400)
        .json({ status: false, error: "Email already registered!" });
    } else {
      console.error(error);
      res.status(500).json({ status: false, error: "Internal server error" });
    }
  }
});

router.get("/subaccounts/:id", (req, res) => {
  const created_by = req.params.id;
  const isSuperAdmin = req.params.id === "3" ? true : false;
  getSubAccounts(created_by, isSuperAdmin, (error, results) => {
    if (error) {
      res.status(500).json({
        status: false,
        error: "An error occurred while fetching data",
      });
    } else {
      res.status(200).json({
        status: true,
        data: results,
      });
    }
  });
});

router.get("/subaccount/:subaccount_id", (req, res) => {
  const subaccount_id = req.params.subaccount_id;
  getSubAccountById(subaccount_id, (error, result) => {
    if (error) {
      res.status(500).json({
        status: false,
        error: "An error occurred while fetching data",
      });
    } else if (result) {
      res.status(200).json({
        status: true,
        data: result,
      });
    } else {
      res.status(404).json({
        status: false,
        error: "Subaccount not found",
      });
    }
  });
});

// router.post("/token", async (req, res) => {
//   const { cid, expire, access_type } = req.body;
//   const missingFields = [];

//   if (!cid) {
//     missingFields.push("cid");
//   }

//   if (!expire) {
//     missingFields.push("expire");
//   }

//   // if (!access_type) {
//   //   missingFields.push("access_type");
//   // }

//   if (missingFields.length > 0) {
//     return res.status(400).json({
//       status: false,
//       error: `Required fields are missing: ${missingFields.join(", ")}`,
//     });
//   }

//   try {
//     const apiUrl = "https://flespi.io/platform/tokens";
//     const headers = {
//       Authorization: `FlespiToken ${process.env.FlespiToken}`,
//       "x-flespi-cid": cid,
//     };
//     const requestBody = [
//       {
//         expire,
//         access: {
//           type: access_type || 0,
//         },
//       },
//     ];

//     const response = await axios.post(apiUrl, requestBody, { headers });

//     if (response.data.result && response.data.result.length > 0) {
//       const insertedData = response.data.result[0];
//       const insertionStatus = await insertSubAccountToken(insertedData);

//       if (insertionStatus) {
//         res.status(response.status).json({ status: true, data: insertedData });
//       } else {
//         res.status(500).json({
//           status: false,
//           error: "Failed to insert data into the database",
//         });
//       }
//     } else {
//       res.status(500).json({
//         status: false,
//         error: "Failed to insert data into the database",
//       });
//     }
//   } catch (error) {
//     if (error.response && error.response.status === 403) {
//       const errorData = error.response.data.errors[0];
//       res.status(403).json({ status: false, error: errorData });
//     } else {
//       console.error(error);
//       res.status(500).json({ status: false, error: "Internal server error" });
//     }
//   }
// });

router.post("/traccar-token", async (req, res) => {
  const { userId, traccar_token } = req.body;

  if (userId && traccar_token) {
    const data = {
      userId,
      traccar_token,
    };
    const result = await insertTraccarToken(data);
    if (result) {
      res.json({ status: true, userId, traccar_token });
    } else {
      res.json({
        status: false,
        error: "Error while inserting or updating data",
      });
    }
  } else {
    res.json({ status: false, error: "Missing or invalid data" });
  }
});

router.get("/traccar-token", (req, res) => {
  getTraccarToken((error, results) => {
    if (error) {
      res.status(500).json({
        status: false,
        error: "An error occurred while fetching data",
      });
    } else {
      res.status(200).json({
        status: true,
        data: results,
      });
    }
  });
});

router.get("/traccar-token/:UserId", (req, res) => {
  const UserId = req.params.UserId;
  getTraccarTokenById(UserId, (error, result) => {
    if (error) {
      res.status(500).json({
        status: false,
        error: "An error occurred while fetching data",
      });
    } else if (result) {
      res.status(200).json({
        status: true,
        data: result,
      });
    } else {
      res.status(404).json({
        status: false,
        error: "Subaccount not found",
      });
    }
  });
});

router.get("/flespi-token/:UserId", (req, res) => {
  const UserId = req.params.UserId;
  getFlespiTokenById(UserId, (error, result) => {
    if (error) {
      res.status(500).json({
        status: false,
        error: "An error occurred while fetching data",
      });
    } else if (result) {
      res.status(200).json({
        status: true,
        data: result,
      });
    } else {
      res.status(404).json({
        status: false,
        error: "Subaccount not found",
      });
    }
  });
});

// ______________________________________________________________________________________________________________________

router.get("/get-session-data", async (req, res) => {
  try {
    const userCookie = req.headers.cookie;
    const response = await axios.get(
      `http://${process.env.TraccarPort}/api/session`,
      {
        headers: {
          Cookie: userCookie,
        },
      }
    );
    res.json(response.data);
  } catch (error) {
    // console.error('Error:', error);
    if (error.response && error.response.data) {
      res.status(500).json({ error: error.response.data });
    } else {
      res.status(500).json({ error: "Internal server error" });
    }
  }
});

// ______________________________________________________________________________________________________________________

router.post("/extract-imeis", async (req, res) => {
  try {
    const data = req.body;
    const uniqueIds = getAllUniqueIds(data);

    const devices = await getDevicesByIMEI(uniqueIds);
    res.json({ status: true, data: devices });
  } catch (error) {
    console.error("Error in /extract-imeis:", error);
    res.status(500).json({ status: false, error: "Internal Server Error" });
  }
});

router.post("/devices-by-imeis", async (req, res) => {
  try {
    const idents = req.body;
    
    const flespiDevices = await axios.get(
      "https://flespi.io/gw/devices/all?fields=id%2Cconfiguration",
      {
        headers: { Authorization: `FlespiToken ${process.env.FlespiToken}` },
      }
    );
      const matchedDevices = getMatchingDevices(idents, flespiDevices.data.result)


    res.status(200).json({ status: true, devices: matchedDevices });
  } catch (error) {
    console.error("Error in /extract-imeis:", error);
    res.status(500).json({ status: false, error: "Internal Server Error" });
  }
});

// ______________________________________________________________________________________________________________________

router.post("/create-device", async (req, res) => {
  const {
    name,
    uniqueId,
    phone,
    model,
    contact,
    category,
    disabled,
    created_by,
  } = req.body;

  if (!name || !uniqueId) {
    return res
      .status(400)
      .json({ status: false, error: "Name and uniqueId are required fields" });
  }

  try {
    const traccarData = {
      name,
      uniqueId,
      ...(phone && { phone }),
      ...(model && { model }),
      ...(contact && { contact }),
      ...(category && { category }),
      disabled,
    };

    const traccarResponse = await axios.post(
      `http://${process.env.TraccarPort}/api/devices`,
      traccarData,
      {
        headers: {
          Authorization: `Bearer ${process.env.TraccarToken}`,
        },
      }
    );

    if (traccarResponse.status === 200) {
      const flespiResponse = await axios.post(
        "https://flespi.io/gw/devices",
        [{ configuration: {}, device_type_id: 0 }],
        {
          headers: {
            Authorization: `FlespiToken ${process.env.FlespiToken}`,
          },
        }
      );

      const responseData = {
        status: true,
        created_by,
        traccarData: traccarResponse.data,
        flespiData: flespiResponse.data.result[0],
      };

      await insertOrUpdateDeviceData([responseData]);
      res.json(responseData);
    } else {
      res.json({ status: true, traccarData: traccarResponse.data });
    }
  } catch (error) {
    if (error.response && error.response.status === 400) {
      res.status(400).json({ status: false, error: error.response.data });
    } else {
      console.error("Error in /create-device:", error);
      res.status(500).json({ status: false, error: "Internal Server Error" });
    }
  }
});

// router.get("/new-devices/:id", (req, res) => {
//   const id = req.params.id;
//   getDeviceData(id, (error, results) => {
//     if (error) {
//       res.status(500).json({
//         status: false,
//         error: "An error occurred while fetching data",
//       });
//     } else {
//       res.status(200).json({
//         status: true,
//         data: results,
//       });
//     }
//   });
// });

// ______________________________________________________________________________________________________________________


module.exports = router;