const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createCommand(data) {
  const sql = `
      INSERT INTO settings_commands (
        userId,
        traccarId,
        deviceId,
        description,
        type,
        attributes
      )
      VALUES ( ?, ?, ?, ?, ?, ?);
    `;

  const values = [
    data.userId,
    data.traccar.id,
    data.traccar.deviceId,
    data.traccar.description,
    data.traccar.type,
    JSON.stringify(data.traccar.attributes),
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getAllCommands() {
  const sql =
    "SELECT id, traccarId, userId, deviceId, description, type, attributes, created_at FROM settings_commands WHERE deleted_at IS NULL;";

  try {
    const result = await dbQuery(sql);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getCommandById(id) {
    const sql =
      "SELECT id, traccarId, userId, deviceId, description, type, attributes, created_at FROM settings_commands WHERE id = ? AND deleted_at IS NULL;";
    const values = [id];
  
    try {
      const result = await dbQuery(sql, values);
  
      if (result.length === 1) {
        const item = result[0];
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      } else {
        return null;
      }
    } catch (error) {
      throw error;
    }
  }  

async function getCommandsByUserId(userId) {
  const sql =
    "SELECT id, traccarId, userId, deviceId, description, type, attributes, created_at FROM settings_commands WHERE userId = ? AND deleted_at IS NULL;";
  const values = [userId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function updateCommandById(id, updatedData) {
  const sql = `
    UPDATE settings_commands
    SET
    deviceId = ?,
    description = ?,
    type = ?,
    attributes = ?
    WHERE id = ?;
  `;

  const values = [
    updatedData.deviceId,
    updatedData.description,
    updatedData.type,
    JSON.stringify(updatedData.attributes),
    id,
  ];

  try {
    const result = await dbQuery(sql, values);
    if (result.affectedRows === 0) {
      throw new Error(`Command with ID ${id} not found.`);
    }

    const updatedresult = {
      id,
      traccar: { ...updatedData },
    };

    return updatedresult;
  } catch (err) {
    throw err;
  }
}

async function softDeleteCommandById(id) {
  const sql = `
        UPDATE settings_commands
        SET
            deleted_at = CURRENT_TIMESTAMP
        WHERE
            id = ?;
    `;

  const values = [id];

  try {
    const result = await dbQuery(sql, values);
    return result.affectedRows;
  } catch (err) {
    throw err;
  }
}

module.exports = {
  createCommand,
  getAllCommands,
  getCommandById,
  getCommandsByUserId,
  updateCommandById,
  softDeleteCommandById,
};
