const db = require("../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createCategory(data) {
  const sql = `
  INSERT INTO settings_categories (name, icon, created_by)
  VALUES (?, ?, ?);
    `;

  const values = [data.name, data.icon, data.created_by];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function updateCategory(categoryId, data) {
  const sql = `
      UPDATE settings_categories
      SET name = ?, icon = ?
      WHERE id = ?;
    `;

  const values = [data.name, data.icon, categoryId];

  try {
    await dbQuery(sql, values);
    return "Category updated successfully";
  } catch (err) {
    throw err;
  }
}

async function deleteCategory(categoryId) {
  const sql = `
      UPDATE settings_categories
      SET deleted_at = CURRENT_TIMESTAMP
      WHERE id = ? AND deleted_at IS NULL;
    `;

  const values = [categoryId];

  try {
    const result = await dbQuery(sql, values);

    if (result.affectedRows === 0) {
      throw new Error("Category not found or already deleted");
    }

    return "Category deleted successfully";
  } catch (err) {
    throw err;
  }
}

async function getCategoryById(categoryId) {
  const sql = `
      SELECT * FROM settings_categories
      WHERE deleted_at IS NULL AND id = ?;
    `;

  const values = [categoryId];

  try {
    const categories = await dbQuery(sql, values);

    if (categories.length === 0) {
      throw new Error("Category not found");
    }

    return categories[0];
  } catch (err) {
    throw err;
  }
}

async function getCategories() {
  const sql = `
      SELECT * FROM settings_categories
      WHERE deleted_at IS NULL
      ORDER BY created_at DESC;
    `;

  try {
    const categories = await dbQuery(sql);
    return categories;
  } catch (err) {
    throw err;
  }
}

async function getCategoriesByCreatedBy(createdBy) {
  const sql = `
      SELECT id, name, icon, created_at FROM settings_categories
      WHERE created_by = ? AND deleted_at IS NULL
      ORDER BY created_at DESC;
    `;

  const values = [createdBy];

  try {
    const categories = await dbQuery(sql, values);
    return categories;
  } catch (err) {
    throw err;
  }
}

module.exports = {
  createCategory,
  updateCategory,
  deleteCategory,
  getCategories,
  getCategoryById,
  getCategoriesByCreatedBy,
};
