const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createAttribute(data) {
  const sql = `
      INSERT INTO settings_attributes_computed (
        userId,
        traccarId,
        description,
        attribute,
        expression,
        type
      )
      VALUES ( ?, ?, ?, ?, ?, ?);
    `;

  const values = [
    data.userId,
    data.traccar.id,
    data.traccar.description,
    data.traccar.attribute,
    data.traccar.expression,
    data.traccar.type,
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getAllAttributes() {
  const sql =
    "SELECT id, traccarId, userId, description, attribute, expression, type, created_at FROM settings_attributes_computed WHERE deleted_at IS NULL;";

  try {
    const result = await dbQuery(sql);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getAttributeById(id) {
    const sql =
      "SELECT id, traccarId, userId, description, attribute, expression, type, created_at FROM settings_attributes_computed WHERE id = ? AND deleted_at IS NULL;";
    const values = [id];
  
    try {
      const result = await dbQuery(sql, values);
  
      if (result.length === 1) {
        const item = result[0];
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      } else {
        return null;
      }
    } catch (error) {
      throw error;
    }
  }  

async function getAttributesByUserId(userId) {
  const sql =
    "SELECT id, traccarId, userId, description, attribute, expression, type, created_at FROM settings_attributes_computed WHERE userId = ? AND deleted_at IS NULL;";
  const values = [userId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function updateAttributeById(id, updatedData) {
  const sql = `
    UPDATE settings_attributes_computed
    SET
    description = ?,
    attribute = ?,
    expression = ?,
    type = ?
    WHERE id = ?;
  `;

  const values = [
    updatedData.description,
    updatedData.attribute,
    updatedData.expression,
    updatedData.type,
    id,
  ];

  try {
    const result = await dbQuery(sql, values);
    if (result.affectedRows === 0) {
      throw new Error(`Attribute with ID ${id} not found.`);
    }

    const updatedresult = {
      id,
      traccar: { ...updatedData },
    };

    return updatedresult;
  } catch (err) {
    throw err;
  }
}

async function softDeleteAttributeById(id) {
  const sql = `
        UPDATE settings_attributes_computed
        SET
            deleted_at = CURRENT_TIMESTAMP
        WHERE
            id = ?;
    `;

  const values = [id];

  try {
    const result = await dbQuery(sql, values);
    return result.affectedRows;
  } catch (err) {
    throw err;
  }
}

module.exports = {
  createAttribute,
  getAllAttributes,
  getAttributeById,
  getAttributesByUserId,
  updateAttributeById,
  softDeleteAttributeById,
};
