const express = require("express");
const router = express.Router();
const {
  createUser,
  getUsers,
  getUserById,
  deleteUser,
  updateUser,
  login,
  blockUser,
  unBlockUser,
} = require("../controllers/userController");
const { generateToken, verifyToken } = require("../utils/jwtUtils");
const { JsonWebTokenError } = require("jsonwebtoken");

// _____________________________________________________________________________________________________________________

router.post("/users", async (req, res) => {
  try {
    const data = req.body;
    const insertID = await createUser(data);

    data.id = insertID;
    data.password = null;

    res.status(201).json({
      status: true,
      message: "User created successfully",
      data: data,
    });
  } catch (error) {
    if(error.code === "ER_DUP_ENTRY"){
        res.status(400).json({
          status: false,
          error: 'Email already exist',
        });
    }else if(error.code === "ER_NO_REFERENCED_ROW_2"){
        res.status(401).json({
          status: false,
          error: 'Incorrect Parent',
        });
    }else{
        console.log(error);
        res.status(500).json({
          status: false,
          error: 'Internal Server Error',
        });
    }
  }
});

router.get("/users", async (req, res) => {
  try {
    const results = await getUsers();
    res.status(200).json({
      status: true,
      data: results,
    });
  } catch (error) {
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching data",
    });
  }
});

router.get("/users/:id", async (req, res) => {
  const userId = req.params.id;
  try {
    const result = await getUserById(userId);
    res.status(200).json({
      status: true,
      data: result,
    });
  } catch (error) {
    res.status(500).json({
      status: false,
      error: "An error occurred while fetching user data",
    });
  }
});

router.put("/users/:id", async (req, res) => {
  const userId = req.params.id;
  const updatedData = req.body;
  try {
    await updateUser(userId, updatedData);
    res.status(200).json({
      status: true,
      message: "User updated successfully",
    });
  } catch (error) {
    res.status(500).json({
      status: false,
      error: "An error occurred while updating the user",
    });
  }
});

router.delete("/users/:id", async (req, res) => {
  const userId = req.params.id;
  try {
    await deleteUser(userId);
    res.status(200).json({
      status: true,
      message: "User deleted successfully",
    });
  } catch (error) {
    res.status(500).json({
      status: false,
      error: "An error occurred while deleting the user",
    });
  }
});

router.post("/login", async (req, res) => {
  try {
    const { email, password } = req.body;
    const userId = await login(email, password);

    const token = generateToken(userId);

    res.status(200).json({
      status: true,
      message: "Login successful",
      data: { id: userId, token },
    });
  } catch (error) {
    const status = error.message === "User not found" ? 404 : 401;
    const errorMessage = error.message || "Invalid credentials";

    res.status(status).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.post("/register", async (req, res) => {
  try {
    const data = req.body;
    const insertID = await createUser(data);

    const token = generateToken(insertID);

    res.status(201).json({
      status: true,
      message: "User registered successfully",
      data: { id: insertID, token },
    });
  } catch (error) {
    const status = error.code === "ER_DUP_ENTRY" ? 400 : 500;
    const errorMessage = error.sqlMessage || "Internal Server Error";

    res.status(status).json({
      status: false,
      error: errorMessage,
    });
  }
});

router.post("/block/:id", async (req, res) => {
  try {
    const userId = req.params.id;
    await blockUser(userId);

    res.status(200).json({
      status: true,
      message: "User blocked successfully",
    });
  } catch (error) {
    res.status(500).json({
      status: false,
      error: "An error occurred while blocking the user",
    });
  }
});

router.post("/unblock/:id", async (req, res) => {
  try {
    const userId = req.params.id;
    await unBlockUser(userId);

    res.status(200).json({
      status: true,
      message: "User blocked successfully",
    });
  } catch (error) {
    res.status(500).json({
      status: false,
      error: "An error occurred while blocking the user",
    });
  }
});

router.get("/user", async (req, res) => {
  try {
    const token = req.headers.authorization;

    if (!token) {
      return res.status(401).json({
        status: false,
        error: "Token not provided",
      });
    }

    const decodedToken = verifyToken(token);

    if (!decodedToken) {
      return res.status(401).json({
        status: false,
        error: "Invalid token",
      });
    }

    const userId = decodedToken.userId;

    const userData = await getUserById(userId);

    if (!userData) {
      return res.status(404).json({
        status: false,
        error: "User not found",
      });
    }

    res.status(200).json({
      status: true,
      data: userData,
    });
  } catch (error) {
    if (error instanceof JsonWebTokenError) {
      return res.status(401).json({
        status: false,
        error: "Invalid token signature",
      });
    } else {
      console.error(error);
    }

    res.status(500).json({
      status: false,
      error: "Internal Server Error",
    });
  }
});

// _____________________________________________________________________________________________________________________

module.exports = router;
