const db = require("../config/dbConfig");

const Device = {
  async insertOrUpdateDevice(deviceData) {
    try {
      await insertOrUpdateDeviceData(deviceData);
      // await insertOrUpdateTelemetryData(deviceData);
      await insertOrUpdateCalcsData(deviceData);
      // await insertOrUpdateConfigData(deviceData);
      // await insertOrUpdateStreamsData(deviceData);

      // console.log(`Data inserted or updated successfully for Device ID: ${deviceData.id}`);
    } catch (error) {
      console.error("Error while inserting or updating device data:", error);
    }
  },
};

async function insertOrUpdateDeviceData(deviceData) {
  const sql = `
    INSERT INTO devices (device_id, messages_ttl, protocol_name, blocked, protocol_id, device_type_id, messages_size, connected, media_size, device_type_name, media_ttl, name, last_active, cid, media_rotate, media_blocked, messages_rotate, ident)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ON DUPLICATE KEY UPDATE
      messages_ttl = VALUES(messages_ttl),
      protocol_name = VALUES(protocol_name),
      blocked = VALUES(blocked),
      protocol_id = VALUES(protocol_id),
      last_active = VALUES(last_active);
  `;

  await db.query(sql, [
    deviceData.id,
    deviceData.messages_ttl,
    deviceData.protocol_name,
    deviceData.blocked,
    deviceData.protocol_id,
    deviceData.device_type_id,
    deviceData.messages_size,
    deviceData.connected,
    deviceData.media_size,
    // deviceData.commands,
    deviceData.device_type_name,
    deviceData.media_ttl,
    deviceData.name,
    deviceData.last_active,
    deviceData.cid,
    deviceData.media_rotate,
    deviceData.media_blocked,
    deviceData.messages_rotate,
    deviceData.configuration.ident,
  ]);
}

async function insertOrUpdateTelemetryData(deviceData) {
  const telemetry = deviceData.telemetry;

  const sql = `
    INSERT INTO devices_telemetry (device_id, ident, position_altitude, position_direction, position_latitude, position_longitude, position_satellites, position_speed, peer, timestamp)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ON DUPLICATE KEY UPDATE
      position_altitude = VALUES(position_altitude),
      position_direction = VALUES(position_direction),
      position_latitude = VALUES(position_latitude),
      position_longitude = VALUES(position_longitude),
      position_satellites = VALUES(position_satellites),
      position_speed = VALUES(position_speed),
      peer = VALUES(peer),
      timestamp = VALUES(timestamp);
  `;

  await db.query(sql, [
    deviceData.id,
    telemetry.ident,
    telemetry.position.altitude,
    telemetry.position.direction,
    telemetry.position.latitude,
    telemetry.position.longitude,
    telemetry.position.satellites,
    telemetry.position.speed,
    telemetry.peer,
    telemetry.timestamp,
  ]);
}

async function insertOrUpdateCalcsData(deviceData) {
  for (const calc of deviceData.calcs) {
    // console.log(calc)
    const sql = `
      INSERT INTO devices_calcs (calc_id, device_id, device_name, calc_name)
      VALUES (?, ?, ?, ?)
      ON DUPLICATE KEY UPDATE
      device_name = VALUES(device_name),
      calc_name = VALUES(calc_name);
    `;

    await db.query(sql, [calc.id, deviceData.id, deviceData.name, calc.name]);
  }
}

async function insertOrUpdateConfigData(deviceData) {
  const sql = `
    INSERT INTO devices_config (ident, device_id)
    VALUES (?, ?)
    ON DUPLICATE KEY UPDATE
    ident = VALUES(ident);
  `;

  await db.query(sql, [deviceData.configuration.ident, deviceData.id]);
}

async function insertOrUpdateStreamsData(deviceData) {
  for (const stream of deviceData.streams) {
    const sql = `
      INSERT INTO devices_streams (device_id, stream_id, name)
      VALUES (?, ?, ?)
      ON DUPLICATE KEY UPDATE
      name = IFNULL(VALUES(name), name);
    `;

    await db.query(sql, [deviceData.id, stream.id, stream.name]);
  }
}

function getAllDevices(callback) {
  const devicesSql = "SELECT * FROM devices";
  db.query(devicesSql, (error, devicesResults) => {
    if (error) {
      console.error("Error:", error);
      callback(error, null);
    } else {
      const devicesWithAllData = devicesResults.map((device) => {
        return new Promise((resolve, reject) => {
          // Define the SQL queries for inner fields
          const telemetrySql =
            "SELECT * FROM devices_telemetry WHERE device_id = ?";
          const calcsSql = "SELECT * FROM devices_calcs WHERE device_id = ?";
          const configsSql = "SELECT * FROM devices_config WHERE device_id = ?";
          const streamsSql =
            "SELECT * FROM devices_streams WHERE device_id = ?";

          // Create promises to query inner fields
          const telemetryQuery = new Promise(
            (resolveTelemetry, rejectTelemetry) => {
              db.query(
                telemetrySql,
                [device.device_id],
                (telemetryError, telemetryResults) => {
                  if (telemetryError) {
                    console.error(
                      "Error fetching telemetry data:",
                      telemetryError
                    );
                    rejectTelemetry(telemetryError);
                  } else {
                    resolveTelemetry(telemetryResults);
                  }
                }
              );
            }
          );

          const calcsQuery = new Promise((resolveCalcs, rejectCalcs) => {
            db.query(
              calcsSql,
              [device.device_id],
              (calcsError, calcsResults) => {
                if (calcsError) {
                  console.error("Error fetching calcs data:", calcsError);
                  rejectCalcs(calcsError);
                } else {
                  resolveCalcs(calcsResults);
                }
              }
            );
          });

          const configsQuery = new Promise((resolveConfigs, rejectConfigs) => {
            db.query(
              configsSql,
              [device.device_id],
              (configsError, configsResults) => {
                if (configsError) {
                  console.error(
                    "Error fetching configurations data:",
                    configsError
                  );
                  rejectConfigs(configsError);
                } else {
                  resolveConfigs(configsResults);
                }
              }
            );
          });

          const streamsQuery = new Promise((resolveStreams, rejectStreams) => {
            db.query(
              streamsSql,
              [device.device_id],
              (streamsError, streamsResults) => {
                if (streamsError) {
                  console.error("Error fetching streams data:", streamsError);
                  rejectStreams(streamsError);
                } else {
                  resolveStreams(streamsResults);
                }
              }
            );
          });

          // Resolve all inner field queries
          Promise.all([telemetryQuery, calcsQuery, configsQuery, streamsQuery])
            .then(([telemetryData, calcsData, configsData, streamsData]) => {
              // Assign the inner field data to the device object
              device.telemetry = telemetryData;
              device.calcs = calcsData;
              device.configs = configsData;
              device.streams = streamsData;
              resolve(device);
            })
            .catch((err) => {
              reject(err);
            });
        });
      });

      // Wait for all devicesWithAllData promises to complete
      Promise.all(devicesWithAllData)
        .then((devicesWithAllData) => {
          // Ensure that all fields are included and map results
          const devicesWithAllFields = devicesWithAllData.map((device) => {
            device.telemetry = device.telemetry || [];
            device.calcs = device.calcs || [];
            device.configs = device.configs || [];
            device.streams = device.streams || [];
            return device;
          });
          callback(null, devicesWithAllFields);
        })
        .catch((err) => {
          callback(err, null);
        });
    }
  });
}

function getDevicesByIMEI(uniqueIds) {
  return new Promise((resolve, reject) => {
    if (uniqueIds.length === 0) {
      resolve([]);
      return;
    }

    const imeiQuery = "SELECT device_id, ident FROM devices WHERE ident IN (?)";
    db.query(imeiQuery, [uniqueIds], (error, devices) => {
      if (error) {
        console.error("Error:", error);
        reject(error);
      } else {
        resolve(devices);
      }
    });
  });
}

module.exports = { Device, getAllDevices, getDevicesByIMEI };
