const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createSnapshot(data) {
  const sql = `
      INSERT INTO settings_snapshots (
        userId,
        flespiId,
        snapshot,
        deviceName,
        s3Path
      )
      VALUES (?, ?, ?, ?, ?)
      ON DUPLICATE KEY UPDATE
        userId = VALUES(userId),
        flespiId = VALUES(flespiId),
        snapshot = VALUES(snapshot),
        deviceName = VALUES(deviceName),
        s3Path = VALUES(s3Path);
    `;

  const values = [
    data.userId,
    data.flespiId,
    data.snapshot,
    data.deviceName,
    data.s3Path,
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function createSnapshot2(data) {
  const sql = `
      INSERT INTO settings_snapshots2 (
        deviceId,
        deviceName,
        startTimestamp,
        endTimestamp,
        contaboEndpoint,
        contaboBucketName,
        contaboBucketAddress,
        fileExtension,
        size,
        filePath,
        uploadResult
      )
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      ON DUPLICATE KEY UPDATE
        deviceName = VALUES(deviceName),
        contaboEndpoint = VALUES(contaboEndpoint),
        contaboBucketName = VALUES(contaboBucketName),
        contaboBucketAddress = VALUES(contaboBucketAddress),
        fileExtension = VALUES(fileExtension),
        size = VALUES(size),
        filePath = VALUES(filePath),
        uploadResult = VALUES(uploadResult);
    `;

  const values = [
    data.deviceId,
    data.deviceName,
    data.startTimestamp,
    data.endTimestamp,
    data.contaboEndpoint,
    data.contaboBucketName,
    data.contaboBucketAddress,
    data.fileExtension,
    data.size,
    data.filePath,
    JSON.stringify(data.uploadResult)
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getAllSnapshots() {
  const sql =
    "SELECT id, userId, flespiId, deviceName, snapshot, s3Path, created_at FROM settings_snapshots WHERE deleted_at IS NULL;";

  try {
    const result = await dbQuery(sql);

    if (result.length > 0) {
      const snapshots = result.map((snapshot) => {
        //   snapshot.attributes = JSON.parse(snapshot.attributes);
        return snapshot;
      });

      return snapshots;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getAllSnapshots2() {
  const sql = `
      SELECT 
          id,
          deviceId,
          deviceName,
          startTimestamp,
          endTimestamp,
          contaboEndpoint,
          contaboBucketName,
          contaboBucketAddress,
          fileExtension,
          size,
          filePath,
          uploadResult,
          created_at 
      FROM settings_snapshots2 
      WHERE deleted_at IS NULL;
  `;

  try {
      const result = await dbQuery(sql);

      if (result.length > 0) {
          const snapshots = result.map((snapshot) => {
              snapshot.uploadResult = JSON.parse(snapshot.uploadResult);
              return snapshot;
          });

          return snapshots;
      } else {
          return [];
      }
  } catch (error) {
      throw error;
  }
}

async function getSnapshotById(snapshotId) {
  const sql =
    "SELECT id, userId, flespiId, deviceName, snapshot, s3Path, created_at FROM settings_snapshots WHERE id = ? AND deleted_at IS NULL;";
  const values = [snapshotId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const snapshot = result[0];
      // snapshot.attributes = JSON.parse(snapshot.attributes);

      return snapshot;
    } else {
      return null;
    }
  } catch (error) {
    throw error;
  }
}

async function getSnapshot2ById(snapshotId) {
  const sql = `
      SELECT 
          id,
          deviceId,
          deviceName,
          startTimestamp,
          endTimestamp,
          contaboEndpoint,
          contaboBucketName,
          contaboBucketAddress,
          fileExtension,
          size,
          filePath,
          uploadResult,
          created_at 
      FROM settings_snapshots2 
      WHERE id = ? 
      AND deleted_at IS NULL;
  `;
  const values = [snapshotId];

  try {
      const result = await dbQuery(sql, values);

      if (result.length > 0) {
          const snapshot = result[0];
          snapshot.uploadResult = JSON.parse(snapshot.uploadResult);
          return snapshot;
      } else {
          return null;
      }
  } catch (error) {
      throw error;
  }
}

async function getSnapshotsByUserId(userId) {
  const sql =
    "SELECT id, userId, flespiId, deviceName, snapshot, s3Path, created_at FROM settings_snapshots WHERE userId = ? AND deleted_at IS NULL;";
  const values = [userId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const snapshots = result.map((snapshot) => {
        //   snapshot.attributes = JSON.parse(snapshot.attributes);
        return snapshot;
      });

      return snapshots;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getSnapshotsByFlespiIds(flespiIds) {
  if (!flespiIds) {
    return [];
  }

  const placeholders = flespiIds.map(() => "?").join(",");

  const sql = `
    SELECT id, userId, flespiId, deviceName, snapshot, s3Path, created_at
    FROM settings_snapshots
    WHERE flespiId IN (${placeholders})
      AND deleted_at IS NULL;
  `;

  try {
    const result = await dbQuery(sql, flespiIds);

    if (result.length > 0) {
      const snapshots = result.map((snapshot) => {
        return snapshot;
      });

      return snapshots;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getSnapshots2ByFlespiIds(flespiIds) {
  if (!flespiIds || flespiIds.length === 0) {
      return [];
  }

  const placeholders = flespiIds.map(() => "?").join(",");

  const sql = `
      SELECT 
          id, 
          deviceId, 
          deviceName, 
          startTimestamp, 
          endTimestamp, 
          contaboEndpoint, 
          contaboBucketName, 
          contaboBucketAddress, 
          fileExtension, 
          size, 
          filePath, 
          uploadResult, 
          created_at 
      FROM settings_snapshots2 
      WHERE deviceId IN (${placeholders}) 
          AND deleted_at IS NULL;
  `;

  try {
      const result = await dbQuery(sql, flespiIds);

      if (result.length > 0) {
          const snapshots = result.map((snapshot) => {
              snapshot.uploadResult = JSON.parse(snapshot.uploadResult);
              return snapshot;
          });

          return snapshots;
      } else {
          return [];
      }
  } catch (error) {
      throw error;
  }
}


async function addSnapshotsDevices(data) {
  for (const device of data) {
    const sql = `
  INSERT INTO settings_snapshots_devices (
    id,
    name
  )
  VALUES (?, ?)
  ON DUPLICATE KEY UPDATE
  name = VALUES(name)
`;

    const values = [device.id, device.name];

    try {
      await dbQuery(sql, values);
    } catch (err) {
      console.error(`Error while processing device with id ${device.id}:`, err);
      throw err;
    }
  }
}

async function getAllIdsAndNames() {
  const sql = `
    SELECT id, name
    FROM settings_snapshots_devices
  `;

  try {
    const result = await dbQuery(sql);
    return result;
  } catch (err) {
    console.error("Error while retrieving all IDs and names:", err);
    throw err;
  }
}

async function removeDevicesByIds(deviceIds) {
  if (!Array.isArray(deviceIds) || deviceIds.length === 0) {
    throw new Error("Invalid or empty array of deviceIds");
  }

  const placeholders = Array.from({ length: deviceIds.length }, () => "?").join(
    ", "
  );

  const sql = `
    DELETE FROM settings_snapshots_devices
    WHERE id IN (${placeholders})
  `;

  try {
    await dbQuery(sql, [...deviceIds]);
  } catch (err) {
    console.error("Error while removing devices by IDs:", err);
    throw err;
  }
}

module.exports = {
  createSnapshot,
  createSnapshot2,
  getAllSnapshots,
  getAllSnapshots2,
  getSnapshotById,
  getSnapshot2ById,
  getSnapshotsByUserId,
  getSnapshotsByFlespiIds,
  getSnapshots2ByFlespiIds,
  addSnapshotsDevices,
  getAllIdsAndNames,
  removeDevicesByIds,
};
