const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createNotifications(data) {
  const sql = `
      INSERT INTO settings_notifications (
        userId,
        traccarId,
        type,
        always,
        calendarId,
        commandId,
        attributes,
        notificators
      )
      VALUES ( ?, ?, ?, ?, ?, ?, ?, ?);
    `;

  const values = [
    data.userId,
    data.traccar.id,
    data.traccar.type,
    data.traccar.always !== undefined ? data.traccar.always : false,
    data.traccar.calendarId,
    data.traccar.commandId,
    JSON.stringify(data.traccar.attributes),
    data.traccar.notificators,
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getAllNotifications() {
  const sql =
    "SELECT id, traccarId, userId, type, always, calendarId, commandId, attributes, notificators, created_at FROM settings_notifications WHERE deleted_at IS NULL;";

  try {
    const result = await dbQuery(sql);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getNotificationById(id) {
  const sql =
    "SELECT id, traccarId, userId, type, always, calendarId, commandId, attributes, notificators, created_at FROM settings_notifications WHERE id = ? AND deleted_at IS NULL;";
  const values = [id];

  try {
    const result = await dbQuery(sql, values);

    if (result.length === 1) {
      const item = result[0];
      if (item.attributes) {
        item.attributes = JSON.parse(item.attributes);
      }
      return item;
    } else {
      return null;
    }
  } catch (error) {
    throw error;
  }
}

async function getNotificationsByUserId(userId) {
  const sql =
    "SELECT id, traccarId, userId, type, always, calendarId, commandId, attributes, notificators, created_at FROM settings_notifications WHERE userId = ? AND deleted_at IS NULL;";
  const values = [userId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function updateNotificationById(id, updatedData) {
  const sql = `
    UPDATE settings_notifications
    SET
    type = ?,
    always = ?,
    calendarId = ?,
    commandId = ?,
    notificators = ?,
    attributes = ?
    WHERE id = ?;
  `;

  const values = [
    updatedData.type,
    updatedData.always,
    updatedData.calendarId,
    updatedData.commandId,
    updatedData.notificators,
    JSON.stringify(updatedData.attributes),
    id,
  ];

  try {
    const result = await dbQuery(sql, values);
    if (result.affectedRows === 0) {
      throw new Error(`Notification with ID ${id} not found.`);
    }

    const updatedresult = {
      id,
      traccar: { ...updatedData },
    };

    return updatedresult;
  } catch (err) {
    throw err;
  }
}

async function softDeleteNotificationById(id) {
  const sql = `
        UPDATE settings_notifications
        SET
            deleted_at = CURRENT_TIMESTAMP
        WHERE
            id = ?;
    `;

  const values = [id];

  try {
    const result = await dbQuery(sql, values);
    return result.affectedRows;
  } catch (err) {
    throw err;
  }
}

module.exports = {
  createNotifications,
  getAllNotifications,
  getNotificationById,
  getNotificationsByUserId,
  updateNotificationById,
  softDeleteNotificationById,
};
