const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createTask(data) {
    const sql = `
        INSERT INTO reports_dispatch_tasks (
          user_id,
          device_id,
          address,
          place_id,
          lat,
          lng,
          message,
          pictures,
          status
        )
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
          user_id = VALUES(user_id),
          device_id = VALUES(device_id),
          address = VALUES(address),
          place_id = VALUES(place_id),
          lat = VALUES(lat),
          lng = VALUES(lng),
          message = VALUES(message),
          pictures = VALUES(pictures),
          status = VALUES(status);
      `;
  
    const values = [
      data.user_id,
      data.device_id,
      data.address,
      data.place_id,
      data.lat,
      data.lng,
      data.message,
      data.pictures,
      data.status || 'pending'
    ];
  
    try {
      const result = await dbQuery(sql, values);
      return result.insertId;
    } catch (err) {
      throw err;
    }
  }

  async function getAllTasks() {
    const sql = `SELECT * FROM reports_dispatch_tasks ORDER BY created_at DESC`;
  
    try {
      const tasks = await dbQuery(sql);
      return tasks;
    } catch (err) {
      throw err;
    }
  }

  async function getTaskById(id) {
    const sql = `SELECT * FROM reports_dispatch_tasks WHERE id = ?`;
  
    try {
      const tasks = await dbQuery(sql, [id]);
      return tasks[0]; 
    } catch (err) {
      throw err;
    }
  }
  

  async function getTasksByUserId(userId) {
    const sql = `SELECT * FROM reports_dispatch_tasks WHERE user_id = ? ORDER BY created_at DESC`;
  
    try {
      const tasks = await dbQuery(sql, [userId]);
      return tasks;
    } catch (err) {
      throw err;
    }
  }
  
  async function getTasksByDeviceId(deviceId) {
    const sql = `SELECT * FROM reports_dispatch_tasks WHERE FIND_IN_SET(?, device_id) ORDER BY created_at DESC`;
  
    try {
      const tasks = await dbQuery(sql, [deviceId]);
      return tasks;
    } catch (err) {
      throw err;
    }
  }
  
  async function updateTaskStatusByIdAndDeviceId(id, deviceId, status, description) {
    let sql = `
      UPDATE reports_dispatch_tasks
      SET status = ?,
          updated_at = CURRENT_TIMESTAMP
    `;
    
    const values = [status];
    
    if (status === 'accepted') {
      sql += `, accepted_at = CURRENT_TIMESTAMP, accepted_description = ?`;
      values.push(description);
    } else if (status === 'rejected') {
      sql += `, rejected_at = CURRENT_TIMESTAMP, rejected_description = ?`;
      values.push(description);
    } else if (status === 'completed') {
      sql += `, completed_at = CURRENT_TIMESTAMP, completed_description = ?`;
      values.push(description);
    }
    
    sql += ` WHERE id = ? AND device_id = ? AND (status = 'pending' OR status = 'accepted');`;
    values.push(id, deviceId);
  
    try {
      const result = await dbQuery(sql, values);
      return result.affectedRows > 0;
    } catch (err) {
      throw err;
    }
  }

  async function createTaskSH(data) {
    const sql = `
        INSERT INTO reports_dispatch_sh (
          user_id,
          address,
          place_id,
          lat,
          lng,
          radius
        )
        VALUES (?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
          address = VALUES(address),
          place_id = VALUES(place_id),
          lat = VALUES(lat),
          lng = VALUES(lng),
          radius = VALUES(radius);
      `;
  
    const values = [
      data.user_id,
      data.address,
      data.place_id,
      data.lat,
      data.lng,
      data.radius
    ];
  
    try {
      const result = await dbQuery(sql, values);
      return result.insertId;
    } catch (err) {
      throw err;
    }
  }
  
  async function getTasksSHByUserId(userId) {
    const sql = `SELECT * FROM reports_dispatch_sh WHERE user_id = ? ORDER BY created_at DESC`;
  
    try {
      const tasks = await dbQuery(sql, [userId]);
      return tasks;
    } catch (err) {
      throw err;
    }
  }
  
  
  async function createPointSH(data) {
    const sql = `
        INSERT INTO reports_point_sh (
          user_id,
          search_time,
          address,
          place_id,
          lat,
          lng,
          radius
        )
        VALUES (?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE
          search_time = VALUES(search_time),
          address = VALUES(address),
          place_id = VALUES(place_id),
          lat = VALUES(lat),
          lng = VALUES(lng),
          radius = VALUES(radius);
      `;
  
    const values = [
      data.user_id,
      data.search_time,
      data.address,
      data.place_id,
      data.lat,
      data.lng,
      data.radius
    ];
  
    try {
      const result = await dbQuery(sql, values);
      return result.insertId;
    } catch (err) {
      throw err;
    }
  }

  async function getPointsSHByUserId(userId) {
    const sql = `SELECT * FROM reports_point_sh WHERE user_id = ? ORDER BY created_at DESC`;
  
    try {
      const Points = await dbQuery(sql, [userId]);
      return Points;
    } catch (err) {
      throw err;
    }
  }
  

module.exports = {
  createTask,
  getAllTasks,
  getTaskById,
  getTasksByUserId,
  getTasksByDeviceId,
  updateTaskStatusByIdAndDeviceId,
  createTaskSH,
  getTasksSHByUserId,
  createPointSH,
  getPointsSHByUserId
};
