const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createCalculator(data) {
  const sql = `
      INSERT INTO settings_calculators (
        id,
        userId,
        flespiId,
        cid,
        selectors,
        messages_source,
        name,
        update_period,
        update_delay,
        update_onchange,
        intervals_ttl,
        intervals_rotate,
        counters,
        validate_interval,
        validate_message,
        timezone,
        metadata,
        version
      )
      VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);
    `;

    const values = [
        data.flespi.id,
        data.userId,
        data.flespi.id,
        data.flespi.cid,
        JSON.stringify(data.flespi.selectors),
        JSON.stringify(data.flespi.messages_source),
        data.flespi.name,
        data.flespi.update_period,
        data.flespi.update_delay,
        data.flespi.update_onchange,
        data.flespi.intervals_ttl,
        data.flespi.intervals_rotate,
        JSON.stringify(data.flespi.counters),
        data.flespi.validate_interval,
        data.flespi.validate_message,
        data.flespi.timezone,
        JSON.stringify(data.flespi.metadata),
        data.flespi.version,
      ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getAllCalculators() {
  const sql =
    "SELECT id, userId, flespiId, cid, selectors, messages_source, name, update_period, update_delay, update_onchange, intervals_ttl, intervals_rotate, counters, validate_interval, validate_message, timezone, metadata, version, created_at FROM settings_calculators WHERE deleted_at IS NULL;";

  try {
    const result = await dbQuery(sql);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.messages_source) {
          item.messages_source = JSON.parse(item.messages_source);
        }
        if (item.selectors) {
          item.selectors = JSON.parse(item.selectors);
        }
        if (item.counters) {
          item.counters = JSON.parse(item.counters);
        }
        if (item.metadata) {
          item.metadata = JSON.parse(item.metadata);
        }

        const booleanFields = [
          "update_onchange",
        ];
        booleanFields.forEach(field => {
          if (typeof item[field] === "number") {
            item[field] = item[field] === 1;
          }
        });

        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }      
}

async function getCalculatorById(id) {
    const sql =
      "SELECT id, userId, flespiId, cid, selectors, messages_source, name, update_period, update_delay, update_onchange, intervals_ttl, intervals_rotate, counters, validate_interval, validate_message, timezone, metadata, version, created_at FROM settings_calculators WHERE id = ? AND deleted_at IS NULL;";
    const values = [id];
  
    try {
        const result = await dbQuery(sql, values);
    
        if (result.length > 0) {
          const item = result[0];
          if (item.messages_source) {
            item.messages_source = JSON.parse(item.messages_source);
          }
          if (item.selectors) {
            item.selectors = JSON.parse(item.selectors);
          }
          if (item.counters) {
            item.counters = JSON.parse(item.counters);
          }
          if (item.metadata) {
            item.metadata = JSON.parse(item.metadata);
          }
          const booleanFields = [
            "update_onchange",
          ];
          booleanFields.forEach(field => {
            if (typeof item[field] === "number") {
              item[field] = item[field] === 1;
            }
          });

          return item;
        } else {
          return null;
        }
      } catch (error) {
        throw error;
      }   
  }  

async function getCalculatorsByUserId(userId) {
  const sql =
    "SELECT id, userId, flespiId, cid, selectors, messages_source, name, update_period, update_delay, update_onchange, intervals_ttl, intervals_rotate, counters, validate_interval, validate_message, timezone, metadata, version, created_at FROM settings_calculators WHERE userId = ? AND deleted_at IS NULL;";
  const values = [userId];

  try {
        const result = await dbQuery(sql, values);
      
        if (result.length > 0) {
          const data = result.map((item) => {
            if (item.messages_source) {
              item.messages_source = JSON.parse(item.messages_source);
            }
            if (item.selectors) {
              item.selectors = JSON.parse(item.selectors);
            }
            if (item.counters) {
              item.counters = JSON.parse(item.counters);
            }
            if (item.metadata) {
              item.metadata = JSON.parse(item.metadata);
            }
            const booleanFields = [
              "update_onchange",
            ];
            booleanFields.forEach(field => {
              if (typeof item[field] === "number") {
                item[field] = item[field] === 1;
              }
            });

            return item;
          });
      
          return data;
        } else {
          return [];
        }
      } catch (error) {
        throw error;
      }   
}

async function updateCalculatorById(id, updatedData) {
  const sql = `
  UPDATE settings_calculators
  SET
  selectors = ?,
  messages_source = ?,
  name = ?,
  update_period = ?,
  update_delay = ?,
  update_onchange = ?,
  intervals_ttl = ?,
  intervals_rotate = ?,
  counters = ?,
  validate_interval = ?,
  validate_message = ?,
  timezone = ?,
  metadata = ?,
  version = ?
  WHERE id = ?;
`;


  const values = [
    JSON.stringify(updatedData.selectors),
    JSON.stringify(updatedData.messages_source),
    updatedData.name,
    updatedData.update_period,
    updatedData.update_delay,
    updatedData.update_onchange,
    updatedData.intervals_ttl,
    updatedData.intervals_rotate,
    JSON.stringify(updatedData.counters),
    updatedData.validate_interval,
    updatedData.validate_message,
    updatedData.timezone,
    JSON.stringify(updatedData.metadata),
    updatedData.version,
    id,
  ];

  try {
    const result = await dbQuery(sql, values);
    if (result.affectedRows === 0) {
      throw new Error(`Calculator with ID ${id} not found.`);
    }

    const updatedresult = {
      id,
      traccar: { ...updatedData },
    };

    return updatedresult;
  } catch (err) {
    throw err;
  }
}

async function softDeleteCalculatorById(id) {
  const sql = `
        UPDATE settings_calculators
        SET
            deleted_at = CURRENT_TIMESTAMP
        WHERE
            id = ?;
    `;

  const values = [id];

  try {
    const result = await dbQuery(sql, values);
    return result.affectedRows;
  } catch (err) {
    throw err;
  }
}

module.exports = {
  createCalculator,
  getAllCalculators,
  getCalculatorById,
  getCalculatorsByUserId,
  updateCalculatorById,
  softDeleteCalculatorById,
};
