const db = require("../config/dbConfig");
const util = require("util");
const { hashPassword, comparePasswords } = require("../helper/common");
const dbQuery = util.promisify(db.query).bind(db);

async function createUser(data) {
  const sql = `
      INSERT INTO users (email, password, admin, super_admin, parent)
      VALUES (?, ?, ?, ?, ?)
    `;

  const hashedPassword = await hashPassword(data.password);

  const values = [
    data.email,
    hashedPassword,
    data.admin || false,
    data.super_admin || false,
    data.parent || null,
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getUsers() {
  const usersSql =
    "SELECT id, email, admin, super_admin, parent, blocked, created_at, updated_at, deleted_at FROM users WHERE deleted_at IS NULL ORDER BY created_at DESC";
  try {
    const usersResults = await dbQuery(usersSql);

    const usersWithParentInfo = await Promise.all(
      usersResults.map(async (user) => {
        const parentSql = "SELECT id, email FROM users WHERE id = ?";
        const parentResult = await dbQuery(parentSql, [user.parent]);
        user.parent = parentResult.length > 0 ? parentResult[0] : null;

        return user;
      })
    );

    return usersWithParentInfo;
  } catch (error) {
    console.log(error);
    throw error;
  }
}

async function getUserById(userId) {
  const userSql =
    "SELECT id, email, admin, super_admin, parent, blocked, created_at, updated_at, deleted_at FROM users WHERE id = ?";
  try {
    const userResult = await dbQuery(userSql, [userId]);
    if (userResult.length === 0) {
      return null;
    }
    const user = userResult[0];
    const parentSql = "SELECT id, parent, email FROM users WHERE id = ?";
    const parentResult = await dbQuery(parentSql, [user.parent]);
    const childrenSql = "SELECT id, parent, email FROM users WHERE parent = ?";
    const childrenResult = await dbQuery(childrenSql, [userId]);
    const grandchildrenSql =
      "SELECT id, parent, email FROM users WHERE parent IN (?)";
    let grandchildResult = [];

    if (childrenResult.length > 0) {
      grandchildResult = await dbQuery(grandchildrenSql, [
        childrenResult.map((child) => child.id),
      ]);
    }

    user.parent = parentResult.length > 0 ? parentResult[0] : null;
    user.childrens = childrenResult;
    user.grandchilds = grandchildResult;

    return user;
  } catch (error) {
    console.log(error);
    throw error;
  }
}

async function deleteUser(userId) {
  const deleteSql =
    "UPDATE users SET deleted_at = CURRENT_TIMESTAMP WHERE id = ?";

  try {
    const result = await dbQuery(deleteSql, [userId]);
    return result;
  } catch (error) {
    throw error;
  }
}

async function updateUser(userId, updatedData) {
  if (updatedData.password) {
    updatedData.password = await hashPassword(updatedData.password);
  }

  const updateSql = "UPDATE users SET ? WHERE id = ?";

  try {
    const result = await dbQuery(updateSql, [updatedData, userId]);
    return result;
  } catch (error) {
    throw error;
  }
}

async function login(email, password) {
  const userSql =
    "SELECT id, password, blocked, deleted_at FROM users WHERE email = ?";
  const userResult = await dbQuery(userSql, [email]);

  if (userResult.length === 0) {
    throw new Error("User not found");
  }

  const user = userResult[0];

  const hashedPassword = user.password;
  const isPasswordMatch = await comparePasswords(password, hashedPassword);

  if (!isPasswordMatch) {
    throw new Error("Incorrect password");
  }

  const userId = user.id;
  const blocked = user.blocked;
  const deletedAt = user.deleted_at;

  if (blocked) {
    throw new Error("User is blocked");
  }

  if (deletedAt) {
    throw new Error("User account is deleted");
  }

  return userId;
}

async function blockUser(userId) {
  const blockSql = "UPDATE users SET blocked = true WHERE id = ?";
  await dbQuery(blockSql, [userId]);
}

async function unBlockUser(userId) {
  const blockSql = "UPDATE users SET blocked = false WHERE id = ?";
  await dbQuery(blockSql, [userId]);
}

module.exports = {
  createUser,
  getUsers,
  getUserById,
  deleteUser,
  updateUser,
  login,
  blockUser,
  unBlockUser,
};
