const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createGeofence(data) {
  const sql = `
      INSERT INTO settings_geofences (
        id,
        userId,
        traccarId,
        name,
        description,
        area,
        calendarId,
        attributes
      )
      VALUES ( ?, ?, ?, ?, ?, ?, ?, ?);
    `;

  const values = [
    data.traccar.id,
    data.userId,
    data.traccar.id,
    data.traccar.name,
    data.traccar.description,
    data.traccar.area,
    data.traccar.calendarId,
    JSON.stringify(data.traccar.attributes),
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getAllGeofences() {
  const sql =
    "SELECT id, traccarId, userId, name, description, area, calendarId, attributes, created_at FROM settings_geofences WHERE deleted_at IS NULL;";

  try {
    const result = await dbQuery(sql);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getGeofenceById(id) {
    const sql =
      "SELECT id, traccarId, userId, name, description, area, calendarId, attributes, created_at FROM settings_geofences WHERE id = ? AND deleted_at IS NULL;";
    const values = [id];
  
    try {
      const result = await dbQuery(sql, values);
  
      if (result.length === 1) {
        const item = result[0];
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      } else {
        return null;
      }
    } catch (error) {
      throw error;
    }
  }  

async function getGeofencesByUserId(userId) {
  const sql =
    "SELECT id, traccarId, userId, name, description, area, calendarId, attributes, created_at FROM settings_geofences WHERE userId = ? AND deleted_at IS NULL;";
  const values = [userId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function updateGeofenceById(id, updatedData) {
  const sql = `
    UPDATE settings_geofences
    SET
    name = ?,
    description = ?,
    area = ?,
    calendarId = ?,
    attributes = ?
    WHERE id = ?;
  `;

  const values = [
    updatedData.name,
    updatedData.description,
    updatedData.area,
    updatedData.calendarId,
    JSON.stringify(updatedData.attributes),
    id,
  ];

  try {
    const result = await dbQuery(sql, values);
    if (result.affectedRows === 0) {
      throw new Error(`Geofence with ID ${id} not found.`);
    }

    const updatedresult = {
      id,
      traccar: { ...updatedData },
    };

    return updatedresult;
  } catch (err) {
    throw err;
  }
}

async function softDeleteGeofenceById(id) {
  const sql = `
        UPDATE settings_geofences
        SET
            deleted_at = CURRENT_TIMESTAMP
        WHERE
            id = ?;
    `;

  const values = [id];

  try {
    const result = await dbQuery(sql, values);
    return result.affectedRows;
  } catch (err) {
    throw err;
  }
}

module.exports = {
  createGeofence,
  getAllGeofences,
  getGeofenceById,
  getGeofencesByUserId,
  updateGeofenceById,
  softDeleteGeofenceById,
};
