const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createDriver(data) {
  const sql = `
      INSERT INTO settings_drivers (
        userId,
        traccarId,
        name,
        uniqueId,
        attributes
      )
      VALUES ( ?, ?, ?, ?, ?);
    `;

  const values = [
    data.userId,
    data.traccar.id,
    data.traccar.name,
    data.traccar.uniqueId,
    JSON.stringify(data.traccar.attributes),
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getAllDrivers() {
  const sql =
    "SELECT id, traccarId, userId, name, uniqueId, attributes, created_at FROM settings_drivers WHERE deleted_at IS NULL;";

  try {
    const result = await dbQuery(sql);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getDriverById(id) {
    const sql =
      "SELECT id, traccarId, userId, name, uniqueId, attributes, created_at FROM settings_drivers WHERE id = ? AND deleted_at IS NULL;";
    const values = [id];
  
    try {
      const result = await dbQuery(sql, values);
  
      if (result.length === 1) {
        const item = result[0];
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      } else {
        return null;
      }
    } catch (error) {
      throw error;
    }
  }  

async function getDriversByUserId(userId) {
  const sql =
    "SELECT id, traccarId, userId, name, uniqueId, attributes, created_at FROM settings_drivers WHERE userId = ? AND deleted_at IS NULL;";
  const values = [userId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function updateDriverById(id, updatedData) {
  const sql = `
    UPDATE settings_drivers
    SET
    name = ?,
    uniqueId = ?,
    attributes = ?
    WHERE id = ?;
  `;

  const values = [
    updatedData.name,
    updatedData.uniqueId,
    JSON.stringify(updatedData.attributes),
    id,
  ];

  try {
    const result = await dbQuery(sql, values);
    if (result.affectedRows === 0) {
      throw new Error(`Driver with ID ${id} not found.`);
    }

    const updatedresult = {
      id,
      traccar: { ...updatedData },
    };

    return updatedresult;
  } catch (err) {
    throw err;
  }
}

async function softDeleteDriverById(id) {
  const sql = `
        UPDATE settings_drivers
        SET
            deleted_at = CURRENT_TIMESTAMP
        WHERE
            id = ?;
    `;

  const values = [id];

  try {
    const result = await dbQuery(sql, values);
    return result.affectedRows;
  } catch (err) {
    throw err;
  }
}

module.exports = {
  createDriver,
  getAllDrivers,
  getDriverById,
  getDriversByUserId,
  updateDriverById,
  softDeleteDriverById,
};
