const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);


async function createDevice(data) {
  const sql = `
      INSERT INTO settings_devices (
        userId,
        password,
        traccarId,
        flespiId,
        name,
        device_type_id,
        uniqueId,
        groupId,
        phone,
        model,
        contact,
        category,
        expirationTime,
        disabled,
        attributes,
        traccar_status,
        traccar_lastUpdate,
        flespi_protocol_name,
        flespi_protocol_id,
        flespi_device_type_name,
        flespi_configuration,
        flespi_metadata,
        media_ttl,
        messages_ttl
      )
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);
    `;

  const values = [
    data.userId,
    data.password,
    data.traccar.id,
    data.flespi.id,
    data.traccar.name,
    data.flespi.device_type_id,
    data.traccar.uniqueId,
    data.traccar.groupId,
    data.traccar.phone,
    data.traccar.model,
    data.traccar.contact,
    data.traccar.category,
    data.traccar.expirationTime,
    data.traccar.disabled,
    JSON.stringify(data.traccar.attributes),
    data.traccar.status,
    data.traccar.lastUpdate,
    data.flespi.protocol_name,
    data.flespi.protocol_id,
    data.flespi.device_type_name,
    JSON.stringify(data.flespi.configuration),
    JSON.stringify(data.flespi.metadata),
    data.flespi.media_ttl,
    data.flespi.messages_ttl,
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getAllDevices() {
  const sql =
    "SELECT id, name, traccarId, flespiId, device_type_id, uniqueId, groupId, phone, model, contact, category, expirationTime, disabled, attributes, userId, traccar_status, traccar_lastUpdate, flespi_protocol_name, flespi_protocol_id, flespi_device_type_name, media_ttl, messages_ttl FROM settings_devices WHERE deleted_at IS NULL;";

  try {
    const result = await dbQuery(sql);

    if (result.length > 0) {
      const devices = result.map((device) => {
        device.attributes = JSON.parse(device.attributes);
        return device;
      });

      return devices;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getDeviceById(deviceId) {
  const sql =
    "SELECT id, name, traccarId, flespiId, device_type_id , uniqueId, groupId, phone, model, contact, category, expirationTime, disabled, attributes, userId, traccar_status, traccar_lastUpdate, flespi_protocol_name, flespi_protocol_id, flespi_device_type_name, media_ttl, messages_ttl FROM settings_devices WHERE id = ? AND deleted_at IS NULL;";
  const values = [deviceId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const device = result[0];
      device.attributes = JSON.parse(device.attributes);

      return device;
    } else {
      return null;
    }
  } catch (error) {
    throw error;
  }
}

async function getDevicesByUserId(userId) {
  const sql =
    "SELECT id, name, traccarId, flespiId, device_type_id , uniqueId, groupId, phone, model, contact, category, expirationTime, disabled, attributes, userId, traccar_status, traccar_lastUpdate, flespi_protocol_name, flespi_protocol_id, flespi_device_type_name, media_ttl, messages_ttl FROM settings_devices WHERE userId = ? AND deleted_at IS NULL;";
  const values = [userId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const devices = result.map((device) => {
        device.attributes = JSON.parse(device.attributes);
        return device;
      });

      return devices;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function updateDeviceById(id, data) {
  let updateSql = `
    UPDATE settings_devices
    SET
      name = ?,
      device_type_id = ?,
      uniqueId = ?,
      groupId = ?,
      phone = ?,
      model = ?,
      contact = ?,
      category = ?,
      expirationTime = ?,
      disabled = ?,
      attributes = ?,
      traccar_status = ?,
      traccar_lastUpdate = ?,
      flespi_protocol_name = ?,
      flespi_protocol_id = ?,
      flespi_device_type_name = ?,
      flespi_configuration = ?,
      media_ttl = ?,
      messages_ttl = ?,
      flespi_metadata = ?
    WHERE id = ?;
  `;

  const selectSql = 'SELECT * FROM settings_devices WHERE id = ?;';
  const values = [
    data.traccar.name,
    data.flespi.device_type_id,
    data.traccar.uniqueId,
    data.traccar.groupId,
    data.traccar.phone,
    data.traccar.model,
    data.traccar.contact,
    data.traccar.category,
    data.traccar.expirationTime,
    data.traccar.disabled,
    JSON.stringify(data.traccar.attributes),
    data.traccar.status,
    data.traccar.lastUpdate,
    data.flespi.protocol_name,
    data.flespi.protocol_id,
    data.flespi.device_type_name,
    JSON.stringify(data.flespi.configuration),
    data.flespi.media_ttl,
    data.flespi.messages_ttl,
    JSON.stringify(data.flespi.metadata),
    id,
  ];

  // Conditionally add password to the update statement
  if (data.password !== undefined && data.password !== null && data.password !== '') {
    updateSql = `
      UPDATE settings_devices
      SET
        password = ?,
        ${updateSql.substr(updateSql.indexOf('SET') + 4)};`;
    values.unshift(data.password);
  }

  try {
    await dbQuery(updateSql, values);
    const result = await dbQuery(selectSql, [id]);

    if (result.length > 0) {
      const updatedRow = result[0];
      updatedRow.flespi_configuration = JSON.parse(updatedRow.flespi_configuration);
      updatedRow.flespi_metadata = JSON.parse(updatedRow.flespi_metadata);
      return updatedRow;
    } else {
      console.log('No rows found with id', id);
      return null;
    }
  } catch (err) {
    throw err;
  }
}

async function softDeleteDeviceById(id) {
    const sql = `
        UPDATE settings_devices
        SET
            deleted_at = CURRENT_TIMESTAMP
        WHERE
            id = ?;
    `;

    const values = [id];

    try {
        const result = await dbQuery(sql, values);
        return result.affectedRows;
    } catch (err) {
        throw err;
    }
}

async function findDeviceByFlespiIdAndPassword(flespiId, password) {
  const selectSql = 'SELECT * FROM settings_devices WHERE flespiId = ? AND password = ?;';
  const values = [flespiId, password];

  try {
    const result = await dbQuery(selectSql, values);

    if (result.length > 0) {
      return result[0];
    } else {
      console.log('No device found with flespiId and password');
      return null;
    }
  } catch (err) {
    throw err;
  }
}

async function findDeviceByUniqueIdAndPassword(uniqueId, password) {
  const selectSql = 'SELECT * FROM settings_devices WHERE uniqueId = ? AND password = ?;';
  const values = [uniqueId, password];

  try {
    const result = await dbQuery(selectSql, values);

    if (result.length > 0) {
      return result[0];
    } else {
      console.log('No device found with uniqueId and password');
      return null;
    }
  } catch (err) {
    throw err;
  }
}

async function getDeviceByTraccarId(deviceId) {
  const sql =
    "SELECT id, name, traccarId, flespiId, device_type_id , uniqueId, groupId, phone, model, contact, category, expirationTime, disabled, attributes, userId, traccar_status, traccar_lastUpdate, flespi_protocol_name, flespi_protocol_id, flespi_device_type_name, media_ttl, messages_ttl FROM settings_devices WHERE traccarId = ? AND deleted_at IS NULL;";
  const values = [deviceId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const device = result[0];
      device.attributes = JSON.parse(device.attributes);

      return device;
    } else {
      return null;
    }
  } catch (error) {
    throw error;
  }
}


  
module.exports = {
  createDevice,
  getAllDevices,
  getDeviceById,
  getDevicesByUserId,
  updateDeviceById,
  softDeleteDeviceById,
  findDeviceByFlespiIdAndPassword,
  findDeviceByUniqueIdAndPassword,
  getDeviceByTraccarId
};
