const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createMaintenance(data) {
  const sql = `
      INSERT INTO settings_maintenance (
        userId,
        traccarId,
        name,
        type,
        start,
        period,
        attributes
      )
      VALUES ( ?, ?, ?, ?, ?, ?, ?);
    `;

  const values = [
    data.userId,
    data.traccar.id,
    data.traccar.name,
    data.traccar.type,
    data.traccar.start,
    data.traccar.period,
    JSON.stringify(data.traccar.attributes),
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getAllMaintenances() {
  const sql =
    "SELECT id, traccarId, userId, name, type, start, period, attributes, created_at FROM settings_maintenance WHERE deleted_at IS NULL;";

  try {
    const result = await dbQuery(sql);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getMaintenanceById(id) {
    const sql =
      "SELECT id, traccarId, userId, name, type, start, period, attributes, created_at FROM settings_maintenance WHERE id = ? AND deleted_at IS NULL;";
    const values = [id];
  
    try {
      const result = await dbQuery(sql, values);
  
      if (result.length === 1) {
        const item = result[0];
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      } else {
        return null;
      }
    } catch (error) {
      throw error;
    }
  }  

async function getMaintenancesByUserId(userId) {
  const sql =
    "SELECT id, traccarId, userId, name, type, start, period, attributes, created_at FROM settings_maintenance WHERE userId = ? AND deleted_at IS NULL;";
  const values = [userId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function updateMaintenanceById(id, updatedData) {
  const sql = `
    UPDATE settings_maintenance
    SET
    name = ?,
    type = ?,
    start = ?,
    period = ?,
    attributes = ?
    WHERE id = ?;
  `;

  const values = [
    updatedData.name,
    updatedData.type,
    updatedData.start,
    updatedData.period,
    JSON.stringify(updatedData.attributes),
    id,
  ];

  try {
    const result = await dbQuery(sql, values);
    if (result.affectedRows === 0) {
      throw new Error(`Maintenance with ID ${id} not found.`);
    }

    const updatedresult = {
      id,
      traccar: { ...updatedData },
    };

    return updatedresult;
  } catch (err) {
    throw err;
  }
}

async function softDeleteMaintenanceById(id) {
  const sql = `
        UPDATE settings_maintenance
        SET
            deleted_at = CURRENT_TIMESTAMP
        WHERE
            id = ?;
    `;

  const values = [id];

  try {
    const result = await dbQuery(sql, values);
    return result.affectedRows;
  } catch (err) {
    throw err;
  }
}

module.exports = {
  createMaintenance,
  getAllMaintenances,
  getMaintenanceById,
  getMaintenancesByUserId,
  updateMaintenanceById,
  softDeleteMaintenanceById,
};
