const db = require("../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createReport(data) {
  const sql = `
    INSERT INTO settings_reports (name, icon, created_by, category_id, devices, calcs, calcs_ids, devices_ids)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?);
  `;

  const values = [
    data.name,
    data.icon,
    data.created_by,
    data.category_id,
    JSON.stringify(data.devices),
    JSON.stringify(data.calcs),
    data.calcs_ids,
    data.devices_ids,
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function updateReport(reportId, data) {
  const sql = `
    UPDATE settings_reports
    SET name = ?, icon = ?, category_id = ?, devices = ?, calcs = ?, calcs_ids = ?, devices_ids = ?
    WHERE id = ?;
  `;

  const values = [
    data.name,
    data.icon,
    data.category_id,
    JSON.stringify(data.devices),
    JSON.stringify(data.calcs),
    data.calcs_ids,
    data.devices_ids,
    reportId,
  ];

  try {
    await dbQuery(sql, values);
    return "Report updated successfully";
  } catch (err) {
    throw err;
  }
}  

async function deleteReport(reportId) {
  const sql = `
    UPDATE settings_reports
    SET deleted_at = CURRENT_TIMESTAMP
    WHERE id = ? AND deleted_at IS NULL;
  `;

  const values = [reportId];

  try {
    const result = await dbQuery(sql, values);

    if (result.affectedRows === 0) {
      throw new Error("Report not found or already deleted");
    }

    return "Report deleted successfully";
  } catch (err) {
    throw err;
  }
}

async function getReportById(reportId) {
  const sql = `
    SELECT * FROM settings_reports
    WHERE deleted_at IS NULL AND id = ?;
  `;

  const values = [reportId];

  try {
    const reports = await dbQuery(sql, values);

    if (reports.length === 0) {
      throw new Error('Report not found');
    }

    const report = reports[0];

    report.devices = report.devices ? JSON.parse(report.devices) : [];
    report.calcs = report.calcs ? JSON.parse(report.calcs) : {};

    return report;
  } catch (err) {
    throw err;
  }
}

async function getReports() {
  const sql = `
    SELECT settings_reports.*, settings_categories.name AS category_name
    FROM settings_reports
    LEFT JOIN settings_categories ON settings_reports.category_id = settings_categories.id
    WHERE settings_reports.deleted_at IS NULL
    ORDER BY settings_reports.created_at DESC;
  `;

  try {
    const reports = await dbQuery(sql);
    
    if (reports.length === 0) {
      throw new Error('Report not found');
    }

    for (const report of reports) {
      report.devices = typeof report.devices === 'string' ? JSON.parse(report.devices) : [];
      report.calcs = typeof report.calcs === 'string' ? JSON.parse(report.calcs) : {};
      report.devices_ids = typeof report.devices_ids === 'string' ? report.devices_ids.split('%2C') : [];
    }

    return reports;
  } catch (err) {
    throw err;
  }
}


async function getReportsByCreatedBy(createdBy) {
  const sql = `
    SELECT settings_reports.*, settings_categories.name AS category_name
    FROM settings_reports
    LEFT JOIN settings_categories ON settings_reports.category_id = settings_categories.id
    WHERE settings_reports.created_by = ? AND settings_reports.deleted_at IS NULL
    ORDER BY settings_reports.created_at DESC;
  `;

  const values = [createdBy];

  try {
    const reports = await dbQuery(sql, values);

    // Modify the response structure
    const modifiedReports = reports.map((report) => {
      return {
        ...report,
        devices_ids: report.devices_ids.split('%2C'),
        devices: JSON.parse(report.devices),
        calcs: JSON.parse(report.calcs),
      };
    });

    return modifiedReports;
  } catch (err) {
    throw err;
  }
}

module.exports = {
  createReport,
  updateReport,
  deleteReport,
  getReports,
  getReportById,
  getReportsByCreatedBy
};
