const bcrypt = require('bcrypt');
const saltRounds = 10;

function extractUniqueDriverIds(data) {
  const driverIds = [];

  data.forEach((item) => {
    const driverIdArray = JSON.parse(item.driver_id);

    if (Array.isArray(driverIdArray)) {
      driverIds.push(...driverIdArray);
    }
  });
  const uniqueDriverIds = [...new Set(driverIds)];
  return uniqueDriverIds;
}

function findAssignedAndNotAssignedDrivers(assignDrivers, allDriverIds) {
  const notAssignedIds = [];
  const assignedDevices = [];

  for (const driverId of allDriverIds) {
    const isAssigned = assignDrivers.some(
      (assignDriver) => assignDriver.rfid_card === driverId
    );

    if (isAssigned) {
      const assignedDriver = assignDrivers.find(
        (assignDriver) => assignDriver.rfid_card === driverId
      );
      assignedDevices.push({
        id: assignedDriver.id,
        driverId: driverId,
        name: assignedDriver.first_name + " " + assignedDriver.last_name,
      });
    } else {
      notAssignedIds.push(driverId);
    }
  }

  return { assignedDevices, notAssignedIds };
}

function transformCanData(data) {
  const transformedData = {};

  for (const key in data) {
    if (typeof data[key] === 'object') {
      const subData = transformCanData(data[key]);
      for (const subKey in subData) {
        transformedData[key.replace(/\./g, '_') + '_' + subKey] = subData[subKey];
      }
    } else {
      transformedData[key.replace(/\./g, '_')] = data[key];
    }
  }

  return transformedData;
}

function transformCanDataArray(dataArray) {
  const transformedArray = [];

  for (const dataItem of dataArray) {
    const transformedData = transformCanData(dataItem);
    transformedArray.push(transformedData);
  }

  return transformedArray;
}

function matchDataWithDeviceCalcs(data, devicesandcalcs) {
  const matchedData = data.map((dataEntry) => {
    const matchingDeviceCalc = devicesandcalcs.find((deviceCalc) => {
      return (
        deviceCalc.calc_id === dataEntry.calc_id &&
        deviceCalc.device_id === dataEntry.device_id
      );
    });

    if (matchingDeviceCalc) {
      return {
        ...dataEntry,
        calc_name: matchingDeviceCalc.calc_name,
        device_name: matchingDeviceCalc.device_name,
      };
    } else {
      return dataEntry;
    }
  });
  return matchedData;
}

function getAllUniqueIds(data) {
  const uniqueIdsArray = Object.values(data).map(item => item.uniqueId);
  return uniqueIdsArray;
}

async function hashPassword(password) {
  try {
    const hashedPassword = await bcrypt.hash(password, saltRounds);
    return hashedPassword;
  } catch (error) {
    throw error;
  }
}

async function comparePasswords(inputPassword, hashedPassword) {
  try {
    const match = await bcrypt.compare(inputPassword, hashedPassword);
    return match;
  } catch (error) {
    throw error;
  }
}

module.exports = {
  extractUniqueDriverIds,
  findAssignedAndNotAssignedDrivers,
  transformCanDataArray,
  matchDataWithDeviceCalcs,
  getAllUniqueIds,
  hashPassword,
  comparePasswords
};
