const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createGroup(data) {
  const sql = `
      INSERT INTO settings_groups (
        traccarId,
        flespiId,
        name,
        attributes,
        groupId,
        userId
      )
      VALUES (?, ?, ?, ?, ?, ?);
    `;

  const values = [
    data.traccar.id,
    data.flespi.id,
    data.traccar.name,
    JSON.stringify(data.traccar.attributes),
    data.traccar.groupId,
    data.userId,
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getAllGroups() {
  const sql =
    "SELECT id, name, traccarId, flespiId, attributes, groupId, userId FROM settings_groups WHERE deleted_at IS NULL;";

  try {
    const result = await dbQuery(sql);

    if (result.length > 0) {
      const groups = result.map((group) => {
        group.attributes = JSON.parse(group.attributes);
        return group;
      });

      return groups;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getGroupById(groupId) {
  const sql =
    "SELECT id, name, traccarId, flespiId, attributes, groupId, userId FROM settings_groups WHERE id = ? AND deleted_at IS NULL;";
  const values = [groupId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const group = result[0];
      group.attributes = JSON.parse(group.attributes);

      return group;
    } else {
      return null;
    }
  } catch (error) {
    throw error;
  }
}

async function getGroupsByUserId(userId) {
  const sql =
    "SELECT id, name, traccarId, flespiId, attributes, groupId, userId FROM settings_groups WHERE userId = ? AND deleted_at IS NULL;";
  const values = [userId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const groups = result.map((group) => {
        group.attributes = JSON.parse(group.attributes);
        return group;
      });

      return groups;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function updateGroupById(id, data) {
    const updateSql = `
        UPDATE settings_groups
        SET
          name = ?,
          attributes = ?,
          groupId = ?
        WHERE id = ?;
    `;

    const selectSql = 'SELECT id, name, traccarId, flespiId, attributes, groupId, userId FROM settings_groups WHERE id = ?;';

    const updateValues = [
        data.traccar.name,
        JSON.stringify(data.traccar.attributes),
        data.traccar.groupId,
        id,
    ];

    try {
        await dbQuery(updateSql, updateValues);

        const result = await dbQuery(selectSql, [id]);

        if (result.length > 0) {
            const updatedRow = result[0];
            // console.log('Updated Row Details:', updatedRow);
            return updatedRow;
        } else {
            console.log('No rows found with id', id);
            return 0;
        }
    } catch (err) {
        throw err;
    }
}

async function softDeleteGroupById(id) {
    const sql = `
        UPDATE settings_groups
        SET
            deleted_at = CURRENT_TIMESTAMP
        WHERE
            id = ?;
    `;

    const values = [id];

    try {
        const result = await dbQuery(sql, values);
        return result.affectedRows;
    } catch (err) {
        throw err;
    }
}
  
module.exports = {
  createGroup,
  updateGroupById,
  getAllGroups,
  getGroupsByUserId,
  getGroupById,
  softDeleteGroupById
};
