const db = require("../../config/dbConfig");
const util = require("util");
const dbQuery = util.promisify(db.query).bind(db);

async function createCalendar(data) {
  const sql = `
      INSERT INTO settings_calendars (
        userId,
        traccarId,
        name,
        data,
        attributes
      )
      VALUES ( ?, ?, ?, ?, ?);
    `;

  const values = [
    data.userId,
    data.traccar.id,
    data.traccar.name,
    data.traccar.data,
    JSON.stringify(data.traccar.attributes),
  ];

  try {
    const result = await dbQuery(sql, values);
    return result.insertId;
  } catch (err) {
    throw err;
  }
}

async function getAllCalendars() {
  const sql =
    "SELECT id, traccarId, userId, name, data, attributes, created_at FROM settings_calendars WHERE deleted_at IS NULL;";

  try {
    const result = await dbQuery(sql);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function getCalendarById(id) {
    const sql =
      "SELECT id, traccarId, userId, name, data, attributes, created_at FROM settings_calendars WHERE id = ? AND deleted_at IS NULL;";
    const values = [id];
  
    try {
      const result = await dbQuery(sql, values);
  
      if (result.length === 1) {
        const item = result[0];
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      } else {
        return null;
      }
    } catch (error) {
      throw error;
    }
  }  

async function getCalendarsByUserId(userId) {
  const sql =
    "SELECT id, traccarId, userId, name, data, attributes, created_at FROM settings_calendars WHERE userId = ? AND deleted_at IS NULL;";
  const values = [userId];

  try {
    const result = await dbQuery(sql, values);

    if (result.length > 0) {
      const data = result.map((item) => {
        if (item.attributes) {
          item.attributes = JSON.parse(item.attributes);
        }
        return item;
      });

      return data;
    } else {
      return [];
    }
  } catch (error) {
    throw error;
  }
}

async function updateCalendarById(id, updatedData) {
  const sql = `
    UPDATE settings_calendars
    SET
    name = ?,
    data = ?,
    attributes = ?
    WHERE id = ?;
  `;

  const values = [
    updatedData.name,
    updatedData.data,
    JSON.stringify(updatedData.attributes),
    id,
  ];

  try {
    const result = await dbQuery(sql, values);
    if (result.affectedRows === 0) {
      throw new Error(`Calendar with ID ${id} not found.`);
    }

    const updatedresult = {
      id,
      traccar: { ...updatedData },
    };

    return updatedresult;
  } catch (err) {
    throw err;
  }
}

async function softDeleteCalendarById(id) {
  const sql = `
        UPDATE settings_calendars
        SET
            deleted_at = CURRENT_TIMESTAMP
        WHERE
            id = ?;
    `;

  const values = [id];

  try {
    const result = await dbQuery(sql, values);
    return result.affectedRows;
  } catch (err) {
    throw err;
  }
}

module.exports = {
  createCalendar,
  getAllCalendars,
  getCalendarById,
  getCalendarsByUserId,
  updateCalendarById,
  softDeleteCalendarById,
};
